package ru.yandex.msearch.util;

import java.io.FilterOutputStream;
import java.io.IOException;
import java.io.OutputStream;

public abstract class BlockCompressingOutputStream
    extends FilterOutputStream
{
    private static final int BUFFER_SIZE = 4096;

    protected byte[] buf;
    protected byte[] compressBuffer;
    protected int count = 0;

    public BlockCompressingOutputStream(
        final OutputStream out,
        final int bufferSize)
    {
        super(out);
        buf = new byte[bufferSize];
        compressBuffer = new byte[bufferSize << 1];
    }

    protected abstract int compress(
        final byte[] src,
        final byte[] dst,
        final int size)
        throws IOException;

    @Override
    public void write(final int b) throws IOException {
        if (count >= buf.length) {
            flushBuffer();
        }
        buf[count++] = (byte) b;
    }

    @Override
    public void write(
        final byte[] b,
        final int off,
        final int len)
        throws IOException
    {
        int remains = len;
        int pos = off;
        int available = buf.length - count;
        while (remains > 0) {
            if (available == 0) {
                flushBuffer();
                available = buf.length;
            }
            int toCopy = Math.min(remains, available);
            System.arraycopy(b, pos, buf, count, toCopy);
            remains -= toCopy;
            available -= toCopy;
            pos += toCopy;
            count += toCopy;
        }
    }

    @Override
    public void flush() throws IOException {
        if (count > 0) {
            flushBuffer();
        }
    }

    private void flushBuffer() throws IOException {
        int compressed = compress(buf, compressBuffer, count);
        writeVInt(compressed);
        writeVInt(count);
//        System.err.println("Compressed: " + compressed + " / " + count);
        count = 0;
        out.write(compressBuffer, 0, compressed);
    }

    private void writeVInt(final int v) throws IOException {
        int i = v;
        while ((i & ~0x7F) != 0) {
            out.write((byte) ((i & 0x7f) | 0x80));
            i >>>= 7;
        }
        out.write((byte) i);
    }
}
