package ru.yandex.msearch.util;

import java.util.AbstractList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.NoSuchElementException;

// Stores contiguous set of bytes, splitting
// data in chunks of 2KB each (2⁹ ints in each chunk)
public class ChunkedByteArray extends AbstractList<Byte> {
    private static final int CHUNK_SIZE_BITS = 11;
    private static final int CHUNK_SIZE = 1 << CHUNK_SIZE_BITS;
    private static final int MASK = CHUNK_SIZE - 1;
    private static final int DEFAULT_INITIAL_CAPACITY = 16;
    private static final byte[] EMPTY_CHUNK = new byte[0];
    private static final byte[][] EMPTY_CHUNKS = new byte[0][];

    private int size = 0;
    private int chunksCount = 0;
    private byte[] lastChunk = EMPTY_CHUNK;
    private int initialChunkCapacity;
    private byte[][] chunks;
    private boolean shrinked = false;

    public ChunkedByteArray() {
        this(0);
    }

    public ChunkedByteArray(final int initialCapacity) {
        if (initialCapacity <= 0) {
            initialChunkCapacity = DEFAULT_INITIAL_CAPACITY;
            chunks = EMPTY_CHUNKS;
        } else {
            initialChunkCapacity = Math.min(initialCapacity, CHUNK_SIZE);
            int chunksRequired = (initialCapacity + MASK) >> CHUNK_SIZE_BITS;
            chunks = new byte[chunksRequired][];
        }
    }

    private void rangeCheck(final int index) {
        if (index >= size) {
            throw new IndexOutOfBoundsException(
                "size: " + size + ", index: " + index);
        }
    }

    public void resize(final int newSize) {
        if (newSize <= size) {
            size = Math.max(0, newSize);
            chunksCount = (size + MASK) >> CHUNK_SIZE_BITS;
            if (chunksCount == 0) {
                lastChunk = EMPTY_CHUNK;
                shrinked = false;
            } else {
                lastChunk = chunks[chunksCount - 1];
                shrinked = (size & MASK) != 0;
            }
        } else {
            int newChunksCount = (newSize + MASK) >> CHUNK_SIZE_BITS;
            int newLastChunkSize = ((newSize - 1) & MASK) + 1;
            if (chunksCount < newChunksCount) {
                if (chunksCount == 0) {
                    chunks = new byte[newChunksCount][];
                } else {
                    if (shrinked) {
                        Arrays.fill(
                            lastChunk,
                            size & MASK,
                            lastChunk.length,
                            (byte) 0);
                        shrinked = false;
                    }
                    chunks[chunksCount - 1] =
                        Arrays.copyOf(lastChunk, CHUNK_SIZE);
                    if (newChunksCount > chunks.length) {
                        chunks = Arrays.copyOf(
                            chunks,
                            Math.max(newChunksCount, chunks.length << 1));
                    }
                }
                while (chunksCount < newChunksCount - 1) {
                    chunks[chunksCount++] = new byte[CHUNK_SIZE];
                }
                lastChunk = new byte[newLastChunkSize];
                chunks[chunksCount++] = lastChunk;
            } else { // chunksCount == newChunksCount && chunksCount > 0
                if (shrinked) {
                    Arrays.fill(
                        lastChunk,
                        size & MASK,
                        lastChunk.length,
                        (byte) 0);
                    shrinked = false;
                }
                lastChunk = Arrays.copyOf(lastChunk, newLastChunkSize);
                chunks[chunksCount - 1] = lastChunk;
            }
            size = newSize;
        }
    }

    @Override
    public void clear() {
        size = 0;
        chunksCount = 0;
        lastChunk = EMPTY_CHUNK;
        chunks = EMPTY_CHUNKS;
        shrinked = false;
    }

    @Override
    public int size() {
        return size;
    }

    @Override
    public boolean add(final Byte element) {
        addByte(element);
        return true;
    }

    public void addByte(final byte element) {
        int pos = size & MASK;
        if (pos == 0) {
            // it is time to add new chunk
            lastChunk = new byte[initialChunkCapacity];
            // next chunk initial capacity should be bigger
            initialChunkCapacity =
                Math.min(initialChunkCapacity << 1, CHUNK_SIZE);
            if (chunksCount == chunks.length) {
                if (chunksCount == 0) {
                    chunks = new byte[DEFAULT_INITIAL_CAPACITY][];
                } else {
                    chunks = Arrays.copyOf(chunks, chunksCount << 1);
                }
            }
            chunks[chunksCount++] = lastChunk;
        } else if (pos == lastChunk.length) {
            // pos < CHUNK_SIZE
            lastChunk =
                Arrays.copyOf(lastChunk, Math.min(pos << 1, CHUNK_SIZE));
            chunks[chunksCount - 1] = lastChunk;
        }
        lastChunk[pos] = element;
        ++size;
    }

    @Override
    public Byte get(final int index) {
        return getByte(index);
    }

    public byte getByte(final int index) {
        rangeCheck(index);
        return chunks[index >> CHUNK_SIZE_BITS][index & MASK];
    }

    @Override
    public Byte set(final int index, final Byte element) {
        return setByte(index, element);
    }

    public byte setByte(final int index, final byte element) {
        rangeCheck(index);
        byte[] chunk = chunks[index >> CHUNK_SIZE_BITS];
        int chunkIndex = index & MASK;
        byte result = chunk[chunkIndex];
        chunk[chunkIndex] = element;
        return result;
    }

    @Override
    public Iterator<Byte> iterator() {
        return new Itr();
    }

    private class Itr implements Iterator<Byte> {
        private int pos = 0;

        @Override
        public boolean hasNext() {
            return pos < size;
        }

        @Override
        public Byte next() {
            if (pos >= size) {
                throw new NoSuchElementException();
            }
            Byte result =
                Byte.valueOf(chunks[pos >> CHUNK_SIZE_BITS][pos & MASK]);
            ++pos;
            return result;
        }
    }
}
