package ru.yandex.msearch.util;

import java.io.IOException;

import java.nio.ByteBuffer;

import ru.yandex.compress.NativeCompress;
import ru.yandex.parser.string.BooleanParser;

import sun.nio.ch.DirectBuffer;

public class Compress extends NativeCompress {
    public static final int IOPRIO_CLASS_BE = 2;
    public static final int IOPRIO_CLASS_IDLE = 3;
    public static final int IOPRIO_HIGH = 0;
    public static final int IOPRIO_LOW = 7;

    public static final boolean QUIET_STDERR =
        BooleanParser.INSTANCE.apply(
            System.getProperty(
                "ru.yandex.lucene.quiet-stderr",
                "false"));
    private static final ThreadLocal<ByteBuffer> tempDirectBuffer =
        new ThreadLocal<>();
    private static final ThreadLocal<IOStats> readStats =
        ThreadLocal.withInitial(IOStats::new);
    private static final ThreadLocal<Boolean> useCache =
        new ThreadLocal<Boolean>();
    private static final ThreadLocal<Boolean> updateSsdCache =
        new ThreadLocal<Boolean>();
    private static final int NATIVE_NICE_MIN = 19;
    private static final int NATIVE_NICE_MAX = -20;
    private static boolean FADVISE_ENABLED = true;

    static {
        tcmallocInit(System.getProperty("java.library.path"));
    }

    public static native boolean tcmallocInit(final String ldLibraryPath);

    private static ByteBuffer getTempDirectBuffer(int size) {
        ByteBuffer bb = tempDirectBuffer.get();
        if (bb == null) {
            bb = ByteBuffer.allocateDirect(size);
            tempDirectBuffer.set(bb);
        } else if (bb.capacity() < size) {
            if (DirectByteBufferCleaner.CLEANER != null) {
                try {
                    DirectByteBufferCleaner.CLEANER.freeBuffer(
                        "preadDirect", bb);
                } catch (IOException ignore) {
                }
            }
            bb = ByteBuffer.allocateDirect(size);
            tempDirectBuffer.set(bb);
        }
        return bb;
    }

    public static int pwrite(
        final int fd,
        final long filePos,
        final long address,
        final int count,
        final boolean fsync)
        throws IOException
    {
        return IOScheduler.instance().writeOp(
            () -> pwrite0(fd, filePos, address, count, fsync));
    }

    public static boolean fadviseEnabled() {
        return FADVISE_ENABLED;
    }

    public static void fadviseEnabled(final boolean enabled) {
        FADVISE_ENABLED = enabled;
    }

    public static long fadviseOp(
        final int fd,
        final long filePos,
        final long length,
        final String tag,
        final boolean countForPread)
        throws IOException
    {
        final long[] readTime = new long[1];
        long time = System.nanoTime();
        IOScheduler.instance().readOp(
            () -> {
                long readStart = System.nanoTime();
                fadvisePrefetch(fd, filePos, length);
                readTime[0] = System.nanoTime() - readStart;
                return 0;
            });
        time = System.nanoTime() - time;
        if (countForPread) {
            readStats.get().read(tag, 0, readTime[0]);
        }
        readStats.get().schedule(tag, time - readTime[0]);
        return readTime[0];
    }

    public static int preadOp(
        final int fd,
        final long filePos,
        final long address,
        final int count,
        final String tag,
        final long timeAdditive)
        throws IOException
    {
        final long[] readTime = new long[1];
        long time = System.nanoTime();
        int ret = IOScheduler.instance().readOp(
            () -> {
                long readStart = System.nanoTime();
                int red = pread(fd, filePos, address, count, tag);
                readTime[0] = System.nanoTime() - readStart;
                return red;
            });
        time = System.nanoTime() - time;
        time += timeAdditive;
        readStats.get().read(tag, count, readTime[0]);
        readStats.get().schedule(tag, time - readTime[0]);
        return ret;
    }

    private static int pread(
        final int fd,
        final long filePos,
        final long address,
        final int count,
        final String tag)
    {
        return pread(fd, filePos, address, count);
    }

    private static int preadDirect(
        final int fd,
        final ByteBuffer dst,
        final long pos,
        final String tag)
    {
        final long address = ((DirectBuffer)dst).address();
        final int bufpos = dst.position();
        final int lim = dst.limit();
        int rem = (bufpos <= lim ? lim - bufpos : 0);
        if (rem == 0) {
            return 0;
        }
        int n = pread(fd, pos, address + bufpos, rem, tag);
        if (n > 0) {
            dst.position(bufpos + n);
        }
        return n;
    }

    public static int pread(
        final int fd,
        final ByteBuffer dst,
        final long pos,
        final String tag)
        throws IOException
    {
        long time = System.nanoTime();
        final long[] readTime = new long[1];
        int ret = IOScheduler.instance().readOp(
            () -> {
                if (dst == null) {
                    throw new NullPointerException();
                }
                if (pos < 0) {
                    throw new IllegalArgumentException("Negative position: "
                        + pos);
                }
                if (dst instanceof DirectBuffer) {
                    return preadDirect(fd, dst, pos, tag);
                }
                ByteBuffer bb = getTempDirectBuffer(dst.remaining());
                bb.rewind();
                bb.limit(dst.remaining());
                long readStart = System.nanoTime();
                int n = preadDirect(fd, bb, pos, tag);
                readTime[0] = System.nanoTime() - readStart;
                bb.flip();
                if (n > 0) {
                    dst.put(bb);
                }
                return n;
            });
        time = System.nanoTime() - time;
        readStats.get().read(tag, ret, readTime[0]);
        readStats.get().schedule(tag, time - readTime[0]);
        return ret;
    }

    public static long readBlockSizes(
        final int fd,
        final long fPos,
        final String tag)
        throws IOException
    {
        long time = System.nanoTime();
        long ret =
            IOScheduler.instance().readOp(() -> readBlockSizes(fd, fPos));
        time = System.nanoTime() - time;
        readStats.get().read(tag, 10, time);
        return ret;
    }

    public static void unpack(
        final String tag,
        final int bytes,
        final long time)
    {
        readStats.get().unpack(tag, bytes, time);
    }

    public static void read(
        final String tag,
        final int bytes,
        final long time)
    {
        readStats.get().read(tag, bytes, time);
    }

    public static void ssdRead(
        final String tag,
        final int bytes,
        final long time)
    {
        readStats.get().ssdRead(tag, bytes, time);
    }

    public static IOStats stats() {
        return readStats.get();
    }

    public static void resetStats() {
        readStats.get().reset();
    }

    public static Boolean useCache() {
        return useCache.get();
    }

    //null: use default behaviour
    //true: force caching
    //false: force directIO
    public void useCache(final Boolean useCache) {
        this.useCache.set(useCache);
    }

    public static boolean updateSsdCache() {
        Boolean update = updateSsdCache.get();
        if (update == null) {
            return false;
        }
        return update;
    }

    public static void updateSsdCache(final boolean updateSsdCache) {
        Compress.updateSsdCache.set(updateSsdCache);
    }

    public static int fastCompressBlock(
        final long input,
        final int inputlen,
        final long output,
        final int outputlen)
    {
        return zstdCompressBlock(input, inputlen, output, outputlen);
    }

    public static int fastDecompressBlock(
        final long input,
        final int inputlen,
        final long output,
        final int outputlen)
    {
        return zstdDecompressBlock(input, inputlen, output, outputlen);
    }

    public static int setThreadPriority(final int prio, boolean mapNice) {
        int p = prio;
        if (mapNice) {
            if (prio == Thread.NORM_PRIORITY) {
                p = 0;
            } else {
                double javaRange =
                    Thread.MAX_PRIORITY - Thread.MIN_PRIORITY + 1;
                double niceRange =
                    NATIVE_NICE_MAX - NATIVE_NICE_MIN - 1;
                double ratio = javaRange / niceRange;
                p = (int) ((prio - Thread.MIN_PRIORITY)
                    / ratio + NATIVE_NICE_MIN);
            }
        }
        int ret = setThreadPriority0(getCurrentThreadId(), p);
        return ret;
    }

    public static int setThreadIOPriority(final int ioclass, final int prio) {
        int ret = setThreadIOPriority0(getCurrentThreadId(), ioclass, prio);
        if (!QUIET_STDERR) {
            System.err.println("setThreadIOPriority0: " + ret);
        }
        return ret;
    }

    public static native int setThreadPriority0(
        final int threadId,
        final int prio);

    public static native int setThreadIOPriority0(
        final int threadId,
        final int ioclass,
        final int prio);

    public static native int getCurrentThreadId();

    public static native void releaseFreeMemory();
//    public static void releaseFreeMemory() {
//    }

    public static native String allocatorStats();
}
