package ru.yandex.msearch.util;

import java.io.IOException;

import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.FutureTask;
import java.util.concurrent.PriorityBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;

import ru.yandex.concurrent.NamedThreadFactory;

import ru.yandex.stater.Stater;
import ru.yandex.stater.StatsConsumer;

public abstract class IOScheduler {
    public static final int IOPRIO_SEARCH = 1000;
    public static final int IOPRIO_JOURNAL = 2000;
    public static final int IOPRIO_INDEXING = 3000;
    public static final int IOPRIO_FLUSH = 4000;
    public static final int IOPRIO_WRITE = 4000;
    public static final int IOPRIO_MERGE = 5000;
    public static final int IOPRIO_WARM = 10000;
    public static final int MAX_PRIO = 0;

    private static final ThreadLocal<Integer> READ_PRIO =
        ThreadLocal.withInitial(() -> new Integer(IOPRIO_MERGE));
    private static final ThreadLocal<Integer> WRITE_PRIO =
        ThreadLocal.withInitial(() -> new Integer(IOPRIO_MERGE));

    private static final IOScheduler DEFAULT = new PassThroughtScheduler();
    private static IOScheduler instance = DEFAULT;
    private static int threads = 2;

    public static IOScheduler instance() {
        return instance;
    }

    public static int getThreadReadPrio() {
        return READ_PRIO.get();
    }

    public static void setThreadReadPrio(final int prio) {
        READ_PRIO.set(prio);
    }

    public static void setThreadWritePrio(final int prio) {
        WRITE_PRIO.set(prio);
    }

    public static void init(final int threads) {
        if (instance == DEFAULT) {
            synchronized (DEFAULT) {
                if (instance == DEFAULT) {
                    instance = new SleepingIOScheduler(threads);
                    instance.start();
                }
            }
        }
    }

    public static void init(
        final int threads,
        final int priorityDivisor,
        final boolean sleeping)
    {
        if (instance == DEFAULT) {
            synchronized (DEFAULT) {
                if (instance == DEFAULT) {
                    instance = new SleepingIOScheduler(
                        threads,
                        priorityDivisor,
                        sleeping);
                    instance.start();
                }
            }
        }
    }

    protected abstract void start();

    public abstract <T> T readOp(final int prio, final Callable<T> op)
        throws IOException;

    public <T> T readOp(final Callable<T> op)
        throws IOException
    {
        return readOp(READ_PRIO.get(), op);
    }

    public abstract <T> T writeOp(final int prio, final Callable<T> op)
        throws IOException;


    public <T> T writeOp(final Callable<T> op)
        throws IOException
    {
        return writeOp(WRITE_PRIO.get(), op);
    }

    public abstract Stater stater(final long metricsTimeFrame);

    private static class PassThroughtScheduler extends IOScheduler {
        @Override
        protected void start() {
        }

        @Override
        public <T> T readOp(final int prio, final Callable<T> op)
            throws IOException
        {
            try {
                return op.call();
            } catch (Exception e) {
                throw new IOException("IO error", e);
            }
        }

        @Override
        public <T> T writeOp(final int prio, final Callable<T> op)
            throws IOException
        {
            return readOp(prio, op);
        }

        public Stater stater(final long metricsTimeFrame) {
            return new Stater() {
                @Override
                public <E extends Exception> void stats(
                    final StatsConsumer<? extends E> statsConsumer)
                    throws E
                {
                }
            };
        }
    }

    private static class ThreadPoolScheduler extends IOScheduler {
        private final int threads;
        private ThreadPoolExecutor iopool;

        public ThreadPoolScheduler(final int threads) {
            this.threads = threads;
        }

        @Override
        protected void start() {
            iopool = new ThreadPoolExecutor(
                threads,
                threads,
                1,
                TimeUnit.DAYS,
                new PriorityBlockingQueue<Runnable>(),
                new NamedThreadFactory("IO") {
                    @Override
                    public Thread newThread(final Runnable r) {
                        return super.newThread(() -> {
                            Compress.setThreadIOPriority(
                                Compress.IOPRIO_CLASS_BE,
                                Compress.IOPRIO_HIGH);
                            r.run();
                        });
                    }
                },
                new ThreadPoolExecutor.CallerRunsPolicy());
        }

        private <T> T ioOp(final int prio, final Callable<T> op)
            throws IOException
        {
            try {
                IOTask<T> task = new IOTask<T>(op, prio);
                iopool.execute(task);
                return task.get();
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new IOException("Interrupted IO", e);
            } catch (ExecutionException e) {
                if (e.getCause() instanceof IOException) {
                    throw (IOException) e.getCause();
                } else {
                    throw new IOException("IO error", e);
                }
            }
        }

        @Override
        public <T> T readOp(final int prio, final Callable<T> op)
            throws IOException
        {
            return ioOp(prio, op);
        }

        @Override
        public <T> T writeOp(final int prio, final Callable<T> op)
            throws IOException
        {
            return ioOp(prio, op);
        }

        public Stater stater(final long metricsTimeFrame) {
            return new Stater() {
                @Override
                public <E extends Exception> void stats(
                    final StatsConsumer<? extends E> statsConsumer)
                    throws E
                {
                }
            };
        }
    }

    private static class IOTask<T> extends FutureTask<T>
        implements Comparable<IOTask<T>>
    {
        private final int priority;
        private final static AtomicLong seq = new AtomicLong();
        private final long seqNum;

        public IOTask(
            final Callable<T> callable,
            final int priority)
        {
            super(callable);
            seqNum = seq.getAndIncrement();
            this.priority = priority;
        }

        @Override
        public void run() {
            try {
                Thread.sleep(5);
                System.err.println("run: " + priority);
                super.run();
            } catch (InterruptedException e) {
            }
        }

        @Override
        public int compareTo(final IOTask<T> other) {
            if (priority < other.priority) {
                return -1;
            } else if (priority > other.priority) {
                return 1;
            } else {
                return Long.compare(seqNum, other.seqNum);
            }
        }
    }
}

