package ru.yandex.msearch.util;

import java.util.function.Function;

import java.util.HashMap;

public class IOStats extends HashMap<String, IOStats> {
    private static final long serialVersionUID = 0L;
    private static final long MILLIS_DIV = 1000000L;
    private static final Function<String, IOStats> MAPPER = x -> new IOStats();

    private final StringBuilder sb = new StringBuilder();
    private int reads = 0;
    private int unpackReads = 0;
    private int schedules = 0;
    private int ssdReads = 0;
    private long totalBytesRead = 0L;
    private long totalBytesUnpacked = 0L;
    private long totalReadTime = 0L;
    private long totalUnpackReadTime = 0L;
    private long totalSSDReadTime = 0L;
    private long totalSSDBytesRead = 0L;
    private long queueTime = 0L;

    public int reads() {
        return reads;
    }

    public int ssdReads() {
        return ssdReads;
    }

    public int unpackReads() {
        return unpackReads;
    }

    public long totalBytesRead() {
        return totalBytesRead;
    }

    public long totalSSDBytesRead() {
        return totalSSDBytesRead;
    }

    public long totalBytesUnpacked() {
        return totalBytesUnpacked;
    }

    public long totalReadTime() {
        return totalReadTime;
    }

    public long totalSSDReadTime() {
        return totalSSDReadTime;
    }

    public long totalUnpackReadTime() {
        return totalUnpackReadTime;
    }

    public long queueTime() {
        return queueTime;
    }

    public void schedule(final String tag, final long time) {
        ++schedules;
        queueTime += time;
        IOStats tagged = computeIfAbsent(tag, MAPPER);
        ++tagged.schedules;
        tagged.queueTime += time;
    }

    public void read(final String tag, final int bytes, final long time) {
        ++reads;
        totalBytesRead += bytes;
        totalReadTime += time;
        IOStats tagged = computeIfAbsent(tag, MAPPER);
        ++tagged.reads;
        tagged.totalBytesRead += bytes;
        tagged.totalReadTime += time;
    }

    public void ssdRead(final String tag, final int bytes, final long time) {
        ++ssdReads;
        totalSSDBytesRead += bytes;
        totalSSDReadTime += time;
        IOStats tagged = computeIfAbsent(tag, MAPPER);
        ++tagged.ssdReads;
        tagged.totalSSDBytesRead += bytes;
        tagged.totalSSDReadTime += time;
    }

    public void unpack(final String tag, final int bytes, final long time) {
        ++unpackReads;
        totalBytesUnpacked += bytes;
        totalUnpackReadTime += time;
        IOStats tagged = computeIfAbsent(tag, MAPPER);
        ++tagged.unpackReads;
        tagged.totalBytesUnpacked += bytes;
        tagged.totalUnpackReadTime += time;
    }

    public void reset() {
        schedules = 0;
        queueTime = 0;
        reads = 0;
        ssdReads = 0;
        unpackReads = 0;
        totalBytesRead = 0;
        totalSSDBytesRead = 0;
        totalBytesUnpacked = 0;
        totalReadTime = 0;
        totalSSDReadTime = 0;
        totalUnpackReadTime = 0;
        clear();
    }

    @Override
    public String toString() {
        sb.setLength(0);
        sb.append('[');
        if (reads > 0) {
            sb.append(" reads: ");
            sb.append(reads);
        }
        if (ssdReads > 0) {
            sb.append(" SSDReads: ");
            sb.append(ssdReads);
        }
        if (unpackReads > 0) {
            sb.append(" reads-unpack: ");
            sb.append(unpackReads);
        }
        if (schedules > 0) {
            sb.append(" waits: ");
            sb.append(schedules);
        }
        if (reads > 0) {
            sb.append(" total-read-time: ");
            sb.append(totalReadTime / MILLIS_DIV);
        }
        if (ssdReads > 0) {
            sb.append(" total-SSD-read-time: ");
            sb.append(totalSSDReadTime / MILLIS_DIV);
        }
        if (unpackReads > 0) {
            sb.append(" total-read-unpack-time: ");
            sb.append(totalUnpackReadTime / MILLIS_DIV);
        }
        if (schedules > 0) {
            sb.append(" total-wait-time: ");
            sb.append(queueTime / MILLIS_DIV);
        }
        if (reads > 0) {
            sb.append(" bytes-read: ");
            sb.append(totalBytesRead);
        }
        if (ssdReads > 0) {
            sb.append(" SSD-bytes-read: ");
            sb.append(totalSSDBytesRead);
        }
        if (unpackReads > 0) {
            sb.append(" bytes-read-unpacked ");
            sb.append(totalBytesUnpacked);
        }
        if (reads > 0) {
            sb.append(" avg-read-time ");
            sb.append(totalReadTime / (MILLIS_DIV * reads));
        }
        if (ssdReads > 0) {
            sb.append(" avg-read-time ");
            sb.append(totalSSDReadTime / (MILLIS_DIV * ssdReads));
        }
        if (unpackReads > 0) {
            sb.append(" avg-read-unpack-time ");
            sb.append(totalUnpackReadTime / (MILLIS_DIV * unpackReads));
        }
        for (Entry<String, IOStats> entry: entrySet()) {
            sb.append(' ');
            sb.append(entry.getKey());
            sb.append(' ');
            sb.append(entry.getValue().toString());
        }
        sb.append(' ');
        sb.append(']');
        return new String(sb);
    }
}
