package ru.yandex.msearch.util;

import java.util.Arrays;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.atomic.LongAdder;

public class JavaAllocator {
    private static final LongAdder TOTAL_SIZE = new LongAdder();
    private static final ConcurrentMap<String, JavaAllocator> POOLS =
        new ConcurrentHashMap<>();
    private static final int PRINT_STATS_DELAY = 5000;
    public static final int LONG_SHIFT = 3;
    public static final int INT_SHIFT = 2;
    public static final int SHORT_SHIFT = 1;

    private static long previousStats = 0;

    private final LongAdder privateSize = new LongAdder();
    private final LongAdder allocs = new LongAdder();
    private final LongAdder reallocs = new LongAdder();
    private final LongAdder deallocs = new LongAdder();

    static {
        Thread t = new Thread("JavaMemoryStatsPrinter") {
            @Override
            public void run() {
                while (true) {
                    try {
                        Thread.sleep(PRINT_STATS_DELAY);
                        printStats();
                    } catch (InterruptedException e) {
                        Thread.currentThread().interrupt();
                    }
                }
            }
        };
        t.setDaemon(true);
        t.start();
    }

    private JavaAllocator() {
    }

    public static JavaAllocator get(final String name) {
        JavaAllocator allocator = POOLS.get(name);
        if (allocator == null) {
            allocator = new JavaAllocator();
            JavaAllocator old = POOLS.putIfAbsent(name, allocator);
            if (old != null) {
                allocator = old;
            }
        }
        return allocator;
    }

    public static Iterator<Map.Entry<String, JavaAllocator>>
        iterator()
    {
        return POOLS.entrySet().iterator();
    }

    public byte[] alloc(final int size) {
        allocs.add(1);
        privateSize.add(size);
        TOTAL_SIZE.add(size);
        return new byte[size];
    }

    public long[] allocLong(final int longSize) {
        final long size = longSize << LONG_SHIFT;
        allocs.add(1);
        privateSize.add(size);
        TOTAL_SIZE.add(size);
        return new long[longSize];
    }

    public int[] allocInt(final int intSize) {
        final long size = intSize << INT_SHIFT;
        allocs.add(1);
        privateSize.add(size);
        TOTAL_SIZE.add(size);
        return new int[intSize];
    }

    public short[] allocShort(final int shortSize) {
        final long size = shortSize << SHORT_SHIFT;
        allocs.add(1);
        privateSize.add(size);
        TOTAL_SIZE.add(size);
        return new short[shortSize];
    }

    public long memorySize() {
        return privateSize.sum();
    }

    public byte[] realloc(final byte[] old, final int newSize) {
        int diff = newSize - old.length;
        TOTAL_SIZE.add(diff);
        privateSize.add(diff);
        reallocs.add(1);
        return Arrays.copyOf(old, newSize);
    }

    public void free(final int size) {
        TOTAL_SIZE.add(-size);
        privateSize.add(-size);
        deallocs.add(1);
    }

    public void free(final byte[] data) {
        TOTAL_SIZE.add(-data.length);
        privateSize.add(-data.length);
        deallocs.add(1);
    }

    public void free(final short[] data) {
        final long size = data.length << SHORT_SHIFT;
        TOTAL_SIZE.add(-size);
        privateSize.add(-size);
        deallocs.add(1);
    }

    public void free(final int[] data) {
        final long size = data.length << INT_SHIFT;
        TOTAL_SIZE.add(-size);
        privateSize.add(-size);
        deallocs.add(1);
    }

    public void free(final long[] data) {
        final long size = data.length << LONG_SHIFT;
        TOTAL_SIZE.add(-size);
        privateSize.add(-size);
        deallocs.add(1);
    }

    public static void printStats() {
        StringBuilder sb = new StringBuilder();
        long stats = TOTAL_SIZE.sum();
        sb.append("\nJavaMemory Stats:");
        sb.append('\n');
        sb.append("\ttotalSize");
        sb.append(':');
        sb.append(' ');
        sb.append(stats);
        sb.append('\n');
        for (Map.Entry<String, JavaAllocator> entry
            : POOLS.entrySet())
        {
            sb.append('\t');
            sb.append(entry.getKey());
            sb.append(':');
            sb.append(' ');
            long tmp = entry.getValue().privateSize.sum();
            stats += tmp;
            sb.append(tmp);
            sb.append(' ');
            sb.append(entry.getValue().allocs.sum());
            sb.append('/');
            sb.append(entry.getValue().reallocs.sum());
            sb.append('/');
            sb.append(entry.getValue().deallocs.sum());
            sb.append('\n');
        }
        if (stats == previousStats) {
            return;
        }
        previousStats = stats;
        System.err.println(sb.toString());
    }
}
