package ru.yandex.msearch.util;

import org.apache.lucene.util.Bits;
import org.apache.lucene.util.OpenBitSet;

import java.util.Arrays;

public class SparseBitSet implements Bits {
    private static final int INITIAL_CAPACITY = 10;
    private static final long DEFAULT_MAX_DISTANCE = 512L;
    private static final OpenBitSet EMPTY_BITSET = new OpenBitSet();

    private final long maxDistance;
    private long[] offsets = new long[INITIAL_CAPACITY];
    private OpenBitSet[] bitsets = new OpenBitSet[INITIAL_CAPACITY];
    private int len = 0;
    private long lastAppendedOffset = Long.MIN_VALUE;
    private OpenBitSet lastAppendedBitSet = EMPTY_BITSET;
    private long lastGetOffset = Long.MAX_VALUE;
    private OpenBitSet lastGetBitSet = null;

    public SparseBitSet() {
        this(DEFAULT_MAX_DISTANCE);
    }

    public SparseBitSet(final long maxDistance) {
        this.maxDistance = maxDistance;
    }

    public void append(final long index) {
        long maxIndex =
            lastAppendedOffset + maxDistance + lastAppendedBitSet.capacity();
        if (maxIndex > index) {
            lastAppendedBitSet.set(index - lastAppendedOffset);
        } else {
            if (len >= offsets.length) {
                int newLen = len << 1;
                long[] offsets = Arrays.copyOf(this.offsets, newLen);
                OpenBitSet[] bitsets = Arrays.copyOf(this.bitsets, newLen);
                this.offsets = offsets;
                this.bitsets = bitsets;
            }
            lastAppendedBitSet = new OpenBitSet();
            lastAppendedBitSet.fastSet(0L);
            lastAppendedOffset = index;
            offsets[len] = index;
            bitsets[len++] = lastAppendedBitSet;
        }
    }

    public boolean get(final long index) {
        // TODO: separate branches, so we can search in left and right parts
        // separately, possibly faster, because index most probably will belong
        // to the next bitset
        if (index < lastGetOffset
            || lastGetOffset + lastGetBitSet.length() <= index)
        {
            int pos = Arrays.binarySearch(offsets, 0, len, index);
            if (pos >= 0) {
                lastGetOffset = offsets[pos];
                lastGetBitSet = bitsets[pos];
                // zero index will be always set
                return true;
            }
            pos = -pos - 1;
            if (pos-- == 0) {
                // Index is somewhere before first bitset
                return false;
            }
            lastGetOffset = offsets[pos];
            lastGetBitSet = bitsets[pos];
        }
        return lastGetBitSet.get(index - lastGetOffset);
    }

    @Override
    public boolean get(final int index) {
        return get((long) index);
    }

    @Override
    public int length() {
        return (int) (lastAppendedOffset + lastAppendedBitSet.length());
    }

    @Override
    public int count() {
        int count = 0;
        for (Bits b: bitsets) {
            count += b.count();
        }
        return count;
    }
}

