package ru.yandex.msearch.util;

import java.util.LinkedHashMap;
import java.util.Map;
import java.util.PriorityQueue;
import java.util.concurrent.PriorityBlockingQueue;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.LockSupport;

public class SumLimiter {
    private final long max;
    private final int bestClients;
    private final long overdraft;
    private final PriorityBlockingQueue<Request> acquireQueue =
        new PriorityBlockingQueue<>();
    private final PriorityBlockingQueue<Request> idleQueue =
        new PriorityBlockingQueue<>();
    private boolean active = true;
    private final AtomicInteger blocked = 
        new AtomicInteger(0);
    private int count = 0;
    private final AtomicInteger waitingIdleLimit =
        new AtomicInteger(0);
    private final AtomicInteger idleCount =
        new AtomicInteger(0);
    private long left;

    public SumLimiter(final long max, final int bestClients) {
        this.max = max;
        this.bestClients = bestClients;
        overdraft = max / bestClients;
        left = max;
    }

    public long max() {
        return max;
    }

    public synchronized long used() {
        return max - left;
    }

    public synchronized int count() {
        return count;
    }

    public synchronized int blocked() {
        return blocked.get();
    }

    public synchronized Map<String, Object> status() {
        Map<String, Object> status = new LinkedHashMap<>();
        status.put("active", active);
        status.put("used", max - left);
        status.put("max", max);
        status.put("best_clients", bestClients);
        status.put("acquired_clients", count);
        status.put("blocked_clients", blocked.get());
        status.put("waiting_idle_limit", waitingIdleLimit.get());
        status.put("idle_clients", idleCount.get());
        return status;
    }

    public synchronized void stop() {
        active = false;
        unparkAll();
    }

    private synchronized boolean limitIdle() {
        if (active && (left < 0L
            || idleCount.get() + count > bestClients))
        {
            return true;
        }
        idleCount.incrementAndGet();
        return false;
    }

    public void waitIdleLimit() {
        waitingIdleLimit.incrementAndGet();
        while (limitIdle()) {
            parkIdle();
        }
        waitingIdleLimit.decrementAndGet();
    }

    public void acquireAfterIdle(final long num) {
        acquire(num);
        idleCount.decrementAndGet();
    }

    public void releaseIdle() {
        idleCount.decrementAndGet();
        unparkIdle();
    }

    private synchronized boolean blockAcquire(final long num) {
        if (active
            && left < num
            && count != 0
            && (count > bestClients || num > overdraft))
        {
            return true;
        }
        count++;
        left -= num;
        return false;
    }

    public void acquire(final long num) {
        blocked.incrementAndGet();
        while (blockAcquire(num)) {
            park(num);
        }
        blocked.decrementAndGet();
    }

    public synchronized void release(final long num) {
        --count;
        left += num;
        unpark(num);
    }

    private void parkIdle() {
        Request request = new Request(Thread.currentThread(), 0);
        idleQueue.add(request);
        request.park();
    }

    private void park(final long size) {
        Request request = new Request(Thread.currentThread(), 0);
        acquireQueue.add(request);
        request.park();
    }

    private void unparkIdle() {
        Request r = idleQueue.peek();
        if (r != null) {
            r = idleQueue.poll();
            r.unpark();
        }
    }

    private void unparkAll() {
        Request r;
        while ((r = idleQueue.poll()) != null) {
            r.unpark();
        }
        while ((r = acquireQueue.poll()) != null) {
            r.unpark();
        }
    }

    private void unpark(final long upto) {
        Request r;
        long toUnpark = upto;
        while ((r = acquireQueue.peek()) != null && toUnpark > 0) {
            if (r.size <= toUnpark) {
                r = acquireQueue.poll();
                r.unpark();
                toUnpark -= r.size();
            } else {
                break;
            }
        }
        if (toUnpark > 0) {
            int unparkIdle = bestClients - (idleCount.get() + count);
            while ((r = idleQueue.peek()) != null && unparkIdle-- > 0) {
                unparkIdle();
            }
        }
    }

    private static final class Request implements Comparable {
        private final Thread thread;
        private final long size;

        public Request(final Thread thread, final long size) {
            this.thread = thread;
            this.size = size;
        }

        @Override
        public int compareTo(final Object o) {
            Request other = (Request)o;
            return Long.compare(size, other.size);
        }

        public void park() {
            LockSupport.park();
        }

        public void unpark() {
            LockSupport.unpark(thread);
        }

        public long size() {
            return this.size;
        }
    }
}
