package ru.yandex.msearch.warmfunction;

import java.text.ParseException;

import java.io.IOException;

import java.util.Arrays;
import java.util.Collection;

import org.apache.lucene.document.FieldMapping;
import org.apache.lucene.document.FieldSelectorResult;
import org.apache.lucene.document.FieldVisitor;
import org.apache.lucene.index.FieldInfo;
import org.apache.lucene.index.FieldInfos;

import org.apache.lucene.index.DocsEnum;
import org.apache.lucene.index.DocsAndPositionsEnum;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.Terms;
import org.apache.lucene.index.TermsEnum;
import org.apache.lucene.index.TermsEnum.SeekStatus;

import org.apache.lucene.search.DocIdSetIterator;

import org.apache.lucene.util.Bits;
import org.apache.lucene.util.BytesRef;

import ru.yandex.msearch.ActivityProvider;

public class KeyIteratorWarmer implements WarmFunction {
    private static final boolean ITERATE_DOCS_DEFAULT = false;
    private static final boolean ITERATE_POSITIONS_DEFAULT = false;
    private static final boolean LOAD_FIELDS_DEFAULT = false;
    private static final boolean REVERSE_DEFAULT = false;
    private static final int LIMIT_DEFAULT = -1;
    private static final int MAX_FREQ_DEFAULT = 100;
    private static final long MAX_INACTIVE_TIME = 7 * 24 * 60 * 60 * 1000;

    private final boolean iterateDocs;
    private final boolean iteratePositions;
    private final boolean loadFields;
    private final int limit;
    private final int maxFreq;
    private final boolean reverse;
    private final FieldVisitor visitor;

    public KeyIteratorWarmer(final String[] args) throws ParseException {
        boolean iterateDocs = ITERATE_DOCS_DEFAULT;
        boolean iteratePositions = ITERATE_POSITIONS_DEFAULT;
        boolean loadFields = LOAD_FIELDS_DEFAULT;
        int limit = LIMIT_DEFAULT;
        int maxFreq = MAX_FREQ_DEFAULT;
        boolean reverse = REVERSE_DEFAULT;

        for (String arg: args) {
            arg = arg.trim();
            int sep = arg.indexOf(' ');
            String name = arg.trim();
            String value = null;
            if (sep != -1) {
                name = arg.substring(0, sep);
                value = arg.substring(sep + 1);
            }
            switch (name) {
                case "docs":
                case "de":
                case "df":
                case "pd":
                case "id":
                case "iterate-docs":
                    iterateDocs = true;
                    break;
                case "prox":
                case "pos":
                case "pp":
                case "positions":
                case "iterate-posision":
                    iteratePositions = true;
                    break;
                case "fields":
                case "load-fields":
                case "lf":
                    loadFields = true;
                    break;
                case "reverse":
                case "desc":
                    reverse = true;
                    break;
                case "limit":
                case "length":
                    limit = parseInt(value, name);
                    break;
                case "max-freq":
                case "maxfreq":
                case "mf":
                    maxFreq = parseInt(value, name);
                    break;
                default:
                    throw new ParseException("Unknown arg: " + name, 0);
            }
        }
        this.iterateDocs = iterateDocs;
        this.iteratePositions = iteratePositions;
        this.loadFields = loadFields;
        if (loadFields) {
            this.visitor = new NullVisitor();
        } else {
            this.visitor = null;
        }
        if (limit == -1) {
            this.limit = Integer.MAX_VALUE;
        } else {
            this.limit = limit;
        }
        this.maxFreq = maxFreq;
        this.reverse = reverse;
        System.err.println("args: " + Arrays.toString(args)
            + ", id: " + iterateDocs
            + ", ip: " + iteratePositions
            + ", lf: " + loadFields
            + ", reverse: " + reverse
            + ", limit: " + limit
            + ", maxFreq: " + maxFreq);
    }

    private int parseInt(String value, String argName) throws ParseException {
        if (value == null) {
            throw new ParseException("Empty arg: " + argName
                + ", number expected", 0);
        }
        try {
            return Integer.parseInt(value);
        } catch (NumberFormatException e) {
            throw new ParseException("Error parsing number for arg: " + argName
                + ", error: " + e.getMessage(), 0);
        }
    }

    @Override
    public void warm(
        final IndexReader reader,
        final Terms terms,
        final Collection<String> prefixes,
        final ActivityProvider activityProvider)
        throws IOException
    {
        TermsEnum te = null;
        if (reverse) {
            try {
                te = terms.reverseIterator();
            } catch (UnsupportedOperationException e) {
                return;
            }
        } else {
            te = terms.iterator(true);
        }
        DocsEnum de = null;
        DocsAndPositionsEnum dpe = null;
        Bits deletedDocs = reader.getDeletedDocs();
        long minTime = System.currentTimeMillis() - MAX_INACTIVE_TIME;
        for (String prefixStr: prefixes) {
            if (!activityProvider.activePrefix(prefixStr)) {
                continue;
            }
            System.err.println("WARMING PREFIX: " + prefixStr);
            BytesRef prefix = new BytesRef(prefixStr);
            SeekStatus ss = te.seek(prefix, false);
            if (ss == SeekStatus.END) {
                System.err.println("SS.END");
                continue;
            }
            int count = 0;
            for (
                BytesRef term = te.term();
                term != null && term.startsWith(prefix) && count < limit;
                term = te.next(), count++)
            {
                if (iteratePositions) {
                    dpe = te.docsAndPositions(deletedDocs, dpe);
                    iteratePositions(dpe, reader);
                } else if (iterateDocs || loadFields) {
                    de = te.docs(deletedDocs, de);
                    iterateDocs(de, reader);
                }
            }
        }
    }

    private void iteratePositions(
        final DocsAndPositionsEnum de,
        final IndexReader reader)
        throws IOException
    {
        for (
            int docId = de.nextDoc(), count = 0;
            docId != DocIdSetIterator.NO_MORE_DOCS && count < maxFreq;
            docId = de.nextDoc(), count++)
        {
            int df = de.freq();
            if (df > 0) {
                de.nextPosition();
            }
            if (loadFields) {
                loadFields(docId, reader);
            }
        }
    }

    private void iterateDocs(
        final DocsEnum de,
        final IndexReader reader)
        throws IOException
    {
        for (
            int docId = de.nextDoc(), count = 0;
            docId != DocIdSetIterator.NO_MORE_DOCS && count < maxFreq;
            docId = de.nextDoc(), count++)
        {
            if (loadFields) {
                loadFields(docId, reader);
            }
        }
    }

    private void loadFields(final int docId, final IndexReader reader)
        throws IOException
    {
//        reader.document(docId);
        reader.readDocument(docId, visitor);
    }

    private static final class NullVisitor implements FieldVisitor {
        private FieldMapping fieldMapping = null;

        @Override
        public FieldMapping fieldMapping(final FieldInfos fieldInfos) {
            if (fieldMapping == null ||
                fieldMapping.fieldSelectorResults().length < fieldInfos.size())
            {
                int size = fieldInfos.size() << 1;
                FieldSelectorResult[] fieldSelectorResults =
                    new FieldSelectorResult[size];
                for (int i = 0; i < size; i++) {
                    fieldSelectorResults[i] =
                        FieldSelectorResult.SIZE_AND_BREAK;
                }
                int[] fieldNumberToIndex = new int[size];
                fieldMapping = new FieldMapping(
                    fieldSelectorResults,
                    fieldNumberToIndex,
                    1);
            }
            return fieldMapping;
        }

        @Override
        public FieldSelectorResult fieldSelectorResult(final String fieldName) {
            return FieldSelectorResult.SIZE_AND_BREAK;
        }

        @Override
        public void storeFieldValue(final int index, final byte[] value) {
        }

        @Override
        public void storeFieldSize(final int index, final int size) {
        }
    }
}

