package ru.yandex.queryParser;

import java.io.IOException;
import java.io.Reader;
import java.io.StringReader;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.analysis.TokenStream;
import org.apache.lucene.analysis.tokenattributes.PositionIncrementAttribute;
import org.apache.lucene.analysis.tokenattributes.TermToBytesRefAttribute;
import org.apache.lucene.index.Term;
import org.apache.lucene.queryParser.ParseException;
import org.apache.lucene.queryParser.QueryParser;
import org.apache.lucene.queryParser.QueryParserBase;
import org.apache.lucene.search.BooleanClause;
import org.apache.lucene.search.BooleanQuery;
import org.apache.lucene.search.MultiPhraseQuery;
import org.apache.lucene.search.MultiTermQuery;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.TermQuery;
import org.apache.lucene.search.PhraseQuery;
import org.apache.lucene.util.BytesRef;
import org.apache.lucene.util.Version;

import ru.yandex.msearch.Config;
import ru.yandex.msearch.FieldConfig;
import ru.yandex.msearch.PrefixingAnalyzerWrapper;
import ru.yandex.msearch.config.DatabaseConfig;

public class YandexQueryParser extends QueryParser {
    protected final DatabaseConfig config;
    private boolean multiFieldQuery = false;
    private int minShould = 0;

    public YandexQueryParser(
        final Version version,
        final String field,
        final Analyzer analyzer,
        final DatabaseConfig config)
    {
        super(version, field, analyzer);
        this.config = config;
    }

    private static Query removePhraseQueries(final Query query) {
        if (query == null || query instanceof MultiPhraseQuery
            || query instanceof PhraseQuery)
        {
            return null;
        }
        if (query instanceof BooleanQuery) {
            BooleanQuery booleanQuery = (BooleanQuery) query;
            Iterator<BooleanClause> iterator = booleanQuery.iterator();
            boolean empty = true;
            while (iterator.hasNext()) {
                BooleanClause clause = iterator.next();
                if (removePhraseQueries(clause.getQuery()) == null) {
                    iterator.remove();
                } else {
                    empty = false;
                }
            }
            if (empty) {
                return null;
            }
        }
        return query;
    }

    @Override
    protected void addClause(final List<BooleanClause> clauses, int conj,
        final int mods, final Query q)
    {
        if (multiFieldQuery) {
            conj = 1;
        }
        super.addClause(clauses, conj, mods, q);
    }

    @Override
    protected BooleanQuery newBooleanQuery(boolean disableCoord) {
        multiFieldQuery = false;
        BooleanQuery bq = super.newBooleanQuery(disableCoord);
        if (minShould > 0) {
            bq.setMinimumNumberShouldMatch(minShould);
        }
        return bq;
    }

    @Override
    protected Query getFieldQuery(final String field, final String text,
        final boolean quoted)
        throws ParseException
    {
        int sep = field.indexOf('@');
        if (sep != -1) {
            minShould = Integer.parseInt(field.substring(sep + 1));
            String configField = field.substring(0, sep);
            return getFieldQuery(configField, configField, text, quoted);
        }
        minShould = 0;
        multiFieldQuery = field.indexOf(',') != -1;
        if (!quoted && multiFieldQuery) {
            QueryParser.Operator operator = getDefaultOperator();
            setDefaultOperator(QueryParser.Operator.AND);
            List<Query> queries = new ArrayList<>();
            try {
                for (String name: field.split(",")) {
                    String value = name.trim();
                    if (!value.isEmpty()) {
                        queries.add(getFieldQuery(value, value, text, quoted));
                    }
                }
            } finally {
                setDefaultOperator(operator);
            }
            return MultiFieldQueryParser.merge(queries);
        } else {
            return getFieldQuery(field, field, text, quoted);
        }
    }

    protected Query getFieldQueryBase(final String field, final String text,
        final boolean quoted)
        throws ParseException
    {
        return super.getFieldQuery(field, text, false);
    }

    protected Query getFieldQuery(final String field, final String configField,
        final String text, final boolean quoted)
        throws ParseException
    {
        Query query = null;
        FieldConfig config = this.config.fieldConfig(configField);
        if (config != null) {
            if (config.attribute()) {
                boolean autoGeneratePhraseQueries =
                    getAutoGeneratePhraseQueries();
                QueryParser.Operator operator = getDefaultOperator();
                setAutoGeneratePhraseQueries(false);
                setDefaultOperator(QueryParser.Operator.AND);
                try {
                    query = removePhraseQueries(
                        getFieldQueryBase(field, text, false));
                } finally {
                    setDefaultOperator(operator);
                    setAutoGeneratePhraseQueries(autoGeneratePhraseQueries);
                }
            } else {
                if (quoted) {
                    query = newQuotedQuery(field, text);
                } else {
                    query = getFieldQueryBase(field, text, false);
                }
            }
        }
        return query;
    }

    @Override
    protected Query getFuzzyQuery(String field, String text,
        float minSimilarity)
        throws ParseException
    {
        if (text.indexOf(' ') == -1) {
            return super.getFuzzyQuery(field, text, minSimilarity);
        } else {
            int slop = (int) minSimilarity;
            Query query = getFieldQuery(field, text, false);
            if (query instanceof PhraseQuery) {
                ((PhraseQuery) query).setSlop(slop);
            } else if (query instanceof MultiPhraseQuery) {
                ((MultiPhraseQuery) query).setSlop(slop);
            }
            return query;
        }
    }

    protected Query newQuotedQuery(final String field, final String text)
        throws ParseException
    {
        try (TokenStream source =
                getAnalyzer().reusableTokenStream(field,
                    new StringReader(text)))
        {
            source.reset();
            if (!source.hasAttribute(TermToBytesRefAttribute.class)) {
                return null;
            }

            TermToBytesRefAttribute termAtt =
                source.getAttribute(TermToBytesRefAttribute.class);

            PositionIncrementAttribute posIncrAtt = null;
            if (source.hasAttribute(PositionIncrementAttribute.class)) {
                posIncrAtt =
                    source.getAttribute(PositionIncrementAttribute.class);
            }

            PhraseQuery query = new PhraseQuery();
            query.setSlop(getPhraseSlop());
            int position = -1;
            int tokens = 0;
            BytesRef term = null;
            while (source.incrementToken()) {
                int positionIncrement = posIncrAtt == null ? 1
                    : posIncrAtt.getPositionIncrement();
                if (positionIncrement > 0) {
                    position += positionIncrement;
                    ++tokens;
                    term = new BytesRef();
                    termAtt.toBytesRef(term);
                    query.add(new Term(field, term), position);
                }
            }
            if (tokens == 1) {
                return newTermQuery(new Term(field, term));
            } else {
                return query;
            }
        } catch (IOException e) {
            // I don't want to know how did you get here...
            ParseException ex = new ParseException("Failed to parse query "
                + field + ':' + text);
            ex.initCause(e);
            throw ex;
        }
    }

    @Override
    protected Query getPrefixQuery(final String field, String termStr) throws ParseException {
        boolean hasLeadingWildcard = termStr.startsWith("*") || termStr.startsWith("?");
        if (hasLeadingWildcard) {
            if (!allowLeadingWildcard) {
                throw new ParseException("'*' not allowed as first character in PrefixQuery");
            } else {
                return super.getPrefixQuery(field, termStr);
            }
        }

        if (lowercaseExpandedTerms) {
            termStr = termStr.toLowerCase();
        }
        if (replaceEExpandedTerms) {
            termStr = termStr.replace('ё', 'е');
        }

            Integer prefixFilterSize = config.prefixSubstringFilterFields().get(field);
        if (prefixFilterSize != null && termStr.length() <= prefixFilterSize) {
            Term t;
            if(analyzer instanceof PrefixingAnalyzerWrapper) {
                PrefixingAnalyzerWrapper pa = (PrefixingAnalyzerWrapper)analyzer;
                if (pa.prefixed(field)) {
                    t = new Term(field, pa.getPrefix() + pa.getSeparator() + termStr);
                } else {
                    t = new Term(field, termStr);
                }
            } else {
                t = new Term(field, termStr);
            }

            return new TermQuery(t);
        } else {
            Term t = new Term(field, termStr);
            return newPrefixQuery(t);
        }
    }
}

