package ru.yandex.search;

import java.io.IOException;

import java.util.Set;

import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.IndexReader.AtomicReaderContext;
import org.apache.lucene.index.Term;
import org.apache.lucene.search.ConstantScoreQuery;
import org.apache.lucene.search.BooleanScorer;
import org.apache.lucene.search.Collector;
import org.apache.lucene.search.IndexSearcher;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.Scorer;
import org.apache.lucene.search.Weight;

public class FieldScoreQueryFilter extends FilterQuery {
    private final String field;
    private final AtomicReaderScoreCollectorFactory collectorFactory;
    private Weight weight = null;

    public FieldScoreQueryFilter(
        final String field,
        final Query in,
        final AtomicReaderScoreCollectorFactory collectorFactory)
    {
        super(in);
        this.field = field;
        this.collectorFactory = collectorFactory;
    }

    @Override
    public String toString(final String field) {
        return "F_" + in.toString(field);
    }

    @Override
    public String toString() {
        return "F_" + in.toString();
    }

    @Override
    public Weight createWeight(final IndexSearcher searcher)
        throws IOException
    {
        return new FieldScoreWeightFilter(in.createWeight(searcher));
    }

    @Override
    public Weight weight(final IndexSearcher searcher) throws IOException {
        return new FieldScoreWeightFilter(in.weight(searcher));
    }

    @Override
    public FieldScoreQueryFilter doClone(final Query query) {
        return new FieldScoreQueryFilter(field, query, collectorFactory);
    }

    @Override
    public int hashCode() {
        return in.hashCode();
    }

    private class FieldScoreWeightFilter extends FilterWeight {
        private final boolean freqSupported;

        public FieldScoreWeightFilter(final Weight in) {
            super(in);
            if (in instanceof ConstantScoreQuery.ConstantWeight) {
                freqSupported = false;
            } else {
                freqSupported = true;
            }
        }

        @Override
        public Scorer scorer(
            final AtomicReaderContext context,
            final ScorerContext scorerContext)
            throws IOException
        {
            final Scorer scorer = in.scorer(context, scorerContext);
            if (scorer != null) {
                final AtomicReaderScoreCollector scoreCollector =
                    collectorFactory.createCollector(context);
                return new FieldScoreFilter(
                    scorer,
                    scoreCollector,
                    freqSupported);
            } else {
                return scorer;
            }
        }
    }

    private class FieldScoreFilter extends FilterScorer {
        private final AtomicReaderScoreCollector scoreCollector;
        private final boolean freqSupported;

        public FieldScoreFilter(
            final Scorer scorer,
            final AtomicReaderScoreCollector scoreCollector)
        {
            this(scorer, scoreCollector, true);
        }

        public FieldScoreFilter(
            final Scorer scorer,
            final AtomicReaderScoreCollector scoreCollector,
            final boolean freqSupported)
        {
            super(scorer);
            this.scoreCollector = scoreCollector;
            this.freqSupported = freqSupported;
        }

        @Override
        public String toString() {
            return "Scorer:" + field + "(" + scoreCollector + ")";
        }

        @Override
        public float score() throws IOException {
            final float score = in.score();
            if (freqSupported) {
                scoreCollector.collect(in.docID(), in.freq(), score);
            } else {
                scoreCollector.collect(in.docID(), 1, score);
            }
            return score;
        }

        @Override
        public void score(final Collector collector) throws IOException {
            in.score(new FieldScoreCollectorFilter(collector, this));
        }

        @Override
        public boolean score(
            final Collector collector,
            final int max,
            final int firstDocID)
            throws IOException
        {
            return in.score(
                new FieldScoreCollectorFilter(collector, this),
                max,
                firstDocID);
        }

        public Scorer scorer() {
            return in;
        }

        public String field() {
            return field;
        }

        public AtomicReaderScoreCollector scoreCollector() {
            return scoreCollector;
        }
    }

    private class FieldScoreCollectorFilter extends FilterCollector {
        private final FieldScoreFilter filterScorer;

        public FieldScoreCollectorFilter(
            final Collector in,
            final FieldScoreFilter filterScorer)
        {
            super(in);
            this.filterScorer = filterScorer;
        }

        @Override
        public void setScorer(final Scorer scorer) throws IOException {
            if (scorer == filterScorer.scorer()) {
                in.setScorer(filterScorer);
            } else {
/*
                if (scorer instanceof BooleanScorer.BucketScorer) {
                    if (filterScorer.scorer().weight() == scorer.weight()) {
                        in.setScorer(
                            new FieldScoreFilter(
                                filterScorer.field(), scorer));
                        return;
                    }
                }
*/
//                if (
//                    (scorer instanceof ConstantScorer)
//                    && (filterScorer.scorer().weight() == scorer.weight()))
                if (filterScorer.scorer().weight() == scorer.weight()) {
                    //Wildcard or Prefix Query
                    in.setScorer(
                        new FieldScoreFilter(
                            scorer,
                            filterScorer.scoreCollector(),
                            false));
                    return;
                }
                System.err.println("Subclassed scorer missmatch: "
                    + "expected: " + filterScorer.scorer()
                    + ", weigth: " + filterScorer.scorer().weight()
                    + ", actual: " + scorer
                    + ", weight: " + scorer.weight());
                in.setScorer(scorer);
                new Exception("setScorer trace").printStackTrace();
            }
        }
    }

    private static class FilterCollector extends Collector {
        protected final Collector in;
        public FilterCollector(final Collector in) {
            this.in = in;
        }

        @Override
        public void setScorer(final Scorer scorer) throws IOException {
            in.setScorer(scorer);
        }

        @Override
        public void collect(int doc) throws IOException {
            in.collect(doc);
        }

        @Override
        public void setNextReader(
            final AtomicReaderContext context)
            throws IOException
        {
            in.setNextReader(context);
        }

        @Override
        public boolean acceptsDocsOutOfOrder() {
            return in.acceptsDocsOutOfOrder();
        }
    }
}
