package ru.yandex.search;

import java.io.IOException;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import org.apache.lucene.search.Scorer;

import ru.yandex.collection.IntList;

import ru.yandex.msearch.collector.YaDoc3;
import ru.yandex.msearch.collector.YaField;

import ru.yandex.search.PerFieldScorerFactory.FieldScorerFactory;
import ru.yandex.search.PerFieldScorerFactory.ScoreType;
import ru.yandex.search.PerFieldScorerFactory.TokenScorerFactory;

public class PerFieldScorer implements YandexScorer {
    public static final String FREQ = "freq";
    public static final String HITS = "hits";
    public static final String SCORE = "score";

    private final Map<String, FieldScorer> perFieldScorers;
    private final YandexScorer[] processors;
    private final ScoreStorage scoreStorage;
    private Scorer scorer = null;

    public PerFieldScorer(
        final Map<String, FieldScorerFactory> perFieldScorersFactory)
    {
        scoreStorage = new ScoreStorage();
        perFieldScorers = new HashMap<>();
        final ArrayList<YandexScorer> processors = new ArrayList<>();
        for (Map.Entry<String, FieldScorerFactory> entry
            : perFieldScorersFactory.entrySet())
        {
            final String field = entry.getKey();
            final FieldScorerFactory factory = entry.getValue();
            final FieldScorer fieldScorer =
                new FieldScorer(field, factory.tokenScorers());
            perFieldScorers.put(field, fieldScorer);
            processors.addAll(fieldScorer.processors());
        }
        this.processors = processors.toArray(new YandexScorer[0]);
    }

    public void setScorer(final Scorer scorer) {
        this.scorer = scorer;
    }

    @Override
    public void preprocess(final int docId)
        throws IOException
    {
        if (scorer != null && !scoreStorage.contains(docId)) {
            scorer.score();
        }
//        if (scorer != null) {
//            scorer.score();
//        }
    }

    @Override
    public void process(final int docId, final YaDoc3 doc)
        throws IOException
    {
        for (final YandexScorer processor : processors) {
            processor.process(docId, doc);
        }
        scoreStorage.removeDocId(docId);
    }

    public FieldScorer fieldScorer(final String field) {
        return perFieldScorers.get(field);
    }

    class FieldScorer {
        private final String field;
        private final Map<
            ScoreType,
            CompoundScoreCollector> perScoreTypeCollectors;
        private YandexScorer[] processors;

        public FieldScorer(
            final String field,
            final Map<ScoreType, Map<String, TokenScorerFactory>> tokenScorers)
        {
            this.field = field;
            perScoreTypeCollectors = new HashMap<>();
            for (Map.Entry<ScoreType, Map<String, TokenScorerFactory>> entry
                : tokenScorers.entrySet())
            {
                final ScoreType type = entry.getKey();
                CompoundScoreCollector collector =
                    perScoreTypeCollectors.get(type);
                if (collector == null) {
                    collector =
                        new CompoundScoreCollector(
                            field,
                            type,
                            scoreStorage);
                    perScoreTypeCollectors.put(type, collector);
                }
                for (Map.Entry<String, TokenScorerFactory> tokenEntry
                    : entry.getValue().entrySet())
                {
                    final TokenScorerFactory tokenScorer =
                        tokenEntry.getValue();
                    final SingleScoreCollector tokenCollector =
                        new SingleScoreCollector(scoreStorage);
                    collector.add(tokenScorer.token(), tokenCollector);
                }
            }
            processors =
                perScoreTypeCollectors.values().toArray(new YandexScorer[0]);
        }

        public AtomicReaderScoreCollector tokenScoreCollector(
            final String token,
            final ScoreType type)
        {
            final CompoundScoreCollector typeCollector =
                perScoreTypeCollectors.get(type);
            if (typeCollector == null) {
                return null;
            }
            return typeCollector.tokenScoreCollector(token);
        }

        public void process(final int docId, final YaDoc3 doc)
            throws IOException
        {
            for (final YandexScorer processor : processors) {
                processor.process(docId, doc);
            }
        }

        public Collection<YandexScorer> processors() {
            return Arrays.<YandexScorer>asList(processors);
        }

    }

    private static class Score {
        public float freq;
        public float score;
        public int hits;

        public Score(final float freq, final float score) {
            this.freq = freq;
            this.score = score;
            hits = 1;
        }
    }

    private static class CompoundScoreCollector implements YandexScorer {
        private final String field;
        private final PerFieldScorerFactory.ScoreType type;
        private final Map<String, SingleScoreCollector> tokenCollectors;
        private final String freqFieldName;
        private final String hitsFieldName;
        private final String scoreFieldName;
        private final IntList collectorsIds;
        private final ScoreStorage scores;

        public CompoundScoreCollector(
            final String field,
            final PerFieldScorerFactory.ScoreType type,
            final ScoreStorage scores)
        {
            this.field = field;
            this.type = type;
            this.scores = scores;
            tokenCollectors = new HashMap<>();
            this.freqFieldName = type.fieldName(field, FREQ);
            this.hitsFieldName = type.fieldName(field, HITS);
            this.scoreFieldName = type.fieldName(field, SCORE);
            collectorsIds = new IntList();
        }

        public void add(
            final String token,
            final SingleScoreCollector collector)
        {
            tokenCollectors.put(token, collector);
            collectorsIds.addInt(collector.scorerId());
        }

        public AtomicReaderScoreCollector tokenScoreCollector(
            final String token)
        {
            return tokenCollectors.get(token);
        }

        @Override
        public void preprocess(final int docId) {
        }

        @Override
        public void process(final int docId, final YaDoc3 doc) {
            final int collectorCount = collectorsIds.size();
            float freq = 0;
            float score = 0;
            int hits = 0;
            final Score[] docScores = scores.docScores(Score.class, docId);
            for (int i = 0; i < collectorCount; i++) {
                final int collectorId = collectorsIds.getInt(i);
                final Score singleScore = docScores[collectorId];
                if (singleScore != null) {
                    freq += singleScore.freq * singleScore.freq;
                    score += singleScore.score;
                    hits += singleScore.hits;
                }
            }
            score = (float) Math.sqrt(score);
            doc.setField(freqFieldName, new YaField.FloatYaField(freq));
            doc.setField(hitsFieldName, new YaField.IntegerYaField(hits));
            doc.setField(scoreFieldName, new YaField.FloatYaField(score));
        }
    }

    private static class SingleScoreCollector
        implements AtomicReaderScoreCollector
    {
        private final int scorerId;
        private final ScoreStorage scores;

        public SingleScoreCollector(final ScoreStorage scores) {
            this.scorerId = scores.newScorerId();
            this.scores = scores;
        }

        public int scorerId() {
            return scorerId;
        }

        @Override
        public void collect(
            final int docId,
            final float freq,
            final float score)
        {
            Score[] docIdScores = scores.docScores(Score.class, docId);
            if (docIdScores[scorerId] == null) {
                docIdScores[scorerId] = new Score(freq, score);
            } else {
                final Score s = docIdScores[scorerId];
                if (s.freq < freq) {
                    s.freq = freq;
                }
                if (s.score < score) {
                    s.score = score;
                }
            }
        }
    }
}
