package ru.yandex.search;

import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Set;

import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.index.IndexReader.AtomicReaderContext;
import org.apache.lucene.index.Term;
import org.apache.lucene.queryParser.ParseException;
import org.apache.lucene.queryParser.QueryParser;
import org.apache.lucene.search.BooleanClause;
import org.apache.lucene.search.BooleanQuery;
import org.apache.lucene.search.MultiPhraseQuery;
import org.apache.lucene.search.MultiTermQuery;
import org.apache.lucene.search.PhraseQuery;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.Scorer;
import org.apache.lucene.search.TermQuery;
import org.apache.lucene.util.StringHelper;
import org.apache.lucene.util.Version;

import ru.yandex.http.util.BadRequestException;

import ru.yandex.msearch.Config;
import ru.yandex.msearch.FieldConfig;

import ru.yandex.msearch.config.DatabaseConfig;
import ru.yandex.queryParser.YandexQueryParser;

public class PerFieldScorerFactory extends YandexScorerFactory {
    private static final int DEFAULT_AUTO_SLOP = 4;

    private final Set<String> outFields;
    private final Map<String, FieldScorerFactory> perFieldScorers;
    private final Map<AtomicReaderContext, PerFieldScorer> contextScorers;

    public PerFieldScorerFactory(final DatabaseConfig config, final String[] args)
        throws BadRequestException
    {
        super(config);
        outFields = new LinkedHashSet<>();
        perFieldScorers = new HashMap<>();
        for (String field : args) {
            final FieldConfig fieldConfig = config.fieldConfig(field);
            if (fieldConfig == null) {
                throw new BadRequestException("Unknown field: " + field);
            }
            if (!perFieldScorers.containsKey(field)) {
                final FieldScorerFactory scorer =
                    new FieldScorerFactory(this, field);
                outFields.addAll(scorer.outFields());
                perFieldScorers.put(field, scorer);
            }
        }
        contextScorers = new HashMap<>();
    }

    @Override
    public QueryParser createParser(
        final Version version,
        final String field,
        final Analyzer analyzer)
    {
        return new PerFieldScoreWrappingQueryParser(
            version,
            field,
            analyzer,
            config,
            perFieldScorers);
    }

    @Override
    public YandexScorer scorer(
        final AtomicReaderContext context,
        final Scorer scorer)
    {
        final PerFieldScorer contextScorer = contextScorer(context);
        contextScorer.setScorer(scorer);
        return contextScorer;
    }

    public PerFieldScorer contextScorer(final AtomicReaderContext context) {
        PerFieldScorer contextScorer = contextScorers.get(context);
        if (contextScorer == null) {
            contextScorer = new PerFieldScorer(perFieldScorers);
            contextScorers.put(context, contextScorer);
        }
        return contextScorer;
    }

    public PerFieldScorer.FieldScorer fieldScorer(
        final String field,
        final AtomicReaderContext context)
    {
        return contextScorer(context).fieldScorer(field);
    }

    @Override
    public Set<String> outFields() {
        return outFields;
    }

    enum ScoreType {
        EXACT,
        NON_EXACT;
//        PHRASE;

        public String fieldName(final String field, final String suffix) {
            return StringHelper.intern(
                '#' + field + '_' + this.toString().toLowerCase()
                + '_' + suffix);
        }
    }

    static class FieldScorerFactory {
        private final PerFieldScorerFactory parent;
        private final String field;
        private final Map<
            ScoreType,
                Map<String, TokenScorerFactory>> perTypePerTokenScorers;

        public FieldScorerFactory(
            final PerFieldScorerFactory parent,
            final String field)
        {
            this.parent = parent;
            this.field = field;
            perTypePerTokenScorers = new HashMap<>();
        }

        public String field() {
            return field;
        }

        public TokenScorerFactory registerToken(
            final String token,
            final ScoreType type)
        {
            Map<String, TokenScorerFactory> perTokenScorers =
                perTypePerTokenScorers.get(type);
            if (perTokenScorers == null) {
                perTokenScorers = new HashMap<>();
                perTypePerTokenScorers.put(type, perTokenScorers);
            }
            TokenScorerFactory scorer = perTokenScorers.get(token);
            if (scorer == null) {
                scorer = new TokenScorerFactory(this, token, type);
                perTokenScorers.put(token, scorer);
            }
            return scorer;
        }

        public Map<ScoreType, Map<String, TokenScorerFactory>> tokenScorers() {
            return perTypePerTokenScorers;
        }

        public Set<String> outFields() {
            final Set<String> outFields = new LinkedHashSet<>();
            for (ScoreType type : ScoreType.values()) {
                outFields.add(type.fieldName(field, PerFieldScorer.FREQ));
                outFields.add(type.fieldName(field, PerFieldScorer.HITS));
                outFields.add(type.fieldName(field, PerFieldScorer.SCORE));
            }
            return outFields;
        }

        public PerFieldScorer.FieldScorer fieldScorer(final AtomicReaderContext context) {
            return parent.fieldScorer(field, context);
        }
    }

    static class TokenScorerFactory
        implements AtomicReaderScoreCollectorFactory
    {
        private final FieldScorerFactory parent;
        private final String token;
        private final ScoreType type;

        public TokenScorerFactory(
            final FieldScorerFactory parent,
            final String token,
            final ScoreType type)
        {
            this.parent = parent;
            this.token = token;
            this.type = type;
        }

        public ScoreType type() {
            return type;
        }

        public String token() {
            return token;
        }

        @Override
        public AtomicReaderScoreCollector createCollector(
            final AtomicReaderContext context)
        {
            final PerFieldScorer.FieldScorer fieldScorer =
                parent.fieldScorer(context);
            if (fieldScorer != null) {
                return fieldScorer.tokenScoreCollector(token, type);
            } else {
                return null;
            }
        }
    }

    private static class PerFieldScoreWrappingQueryParser
        extends YandexQueryParser
    {
        private final Map<String, FieldScorerFactory> perFieldScorers;

        public PerFieldScoreWrappingQueryParser(
            final Version version,
            final String field,
            final Analyzer analyzer,
            final DatabaseConfig config,
            final Map<String, FieldScorerFactory> perFieldScorers)
        {
            super(version, field, analyzer, config);
            this.perFieldScorers = perFieldScorers;
        }

        private TokenScorerFactory getTokenScorer(
            final FieldScorerFactory fieldScorer,
            final String token,
            final ScoreType scoreType)
        {
            return fieldScorer.registerToken(token, scoreType);
        }

        private Query wrapQuery(
            Query query,
            final String field,
            final String text,
            final boolean quoted)
        {
            final FieldScorerFactory fieldScorer = perFieldScorers.get(field);
            if (fieldScorer == null) {
                //compute score only for asked fields
                return query;
            }
            if (query instanceof BooleanQuery) {
                final BooleanQuery origQuery = (BooleanQuery) query;
                query = new BooleanQuery(origQuery.isCoordDisabled());
                ScoreType scoreType = ScoreType.EXACT;
                for (BooleanClause subQuery : origQuery) {
                    //this should effectively return the same scorer
                    //for different tokens at the same position
                    final TokenScorerFactory scorer =
                        getTokenScorer(
                            fieldScorer,
                            text,
                            scoreType);

                    ((BooleanQuery) query).add(
                        new FieldScoreQueryFilter(
                            field,
                            subQuery.getQuery(),
                            scorer),
                        subQuery.getOccur());
                    scoreType = ScoreType.NON_EXACT;
                }
            } else if (query instanceof TermQuery) {
                final ScoreType scoreType =
                    quoted ? ScoreType.EXACT : ScoreType.NON_EXACT;
                final ScoreType fakeScoreType =
                    !quoted ? ScoreType.EXACT : ScoreType.NON_EXACT;
                final TokenScorerFactory scorer =
                    getTokenScorer(
                        fieldScorer,
                        text,
                        scoreType);
                query = new FieldScoreQueryFilter(field, query, scorer);
                //register empty scorer for rendering zeros in outFields
                getTokenScorer(
                    fieldScorer,
                    text,
                    fakeScoreType);
            } else if (query instanceof MultiTermQuery) {
                final TokenScorerFactory scorer =
                    getTokenScorer(
                        fieldScorer,
                        text,
                        ScoreType.NON_EXACT);
                query = new FieldScoreQueryFilter(field, query, scorer);
                //register empty scorer for rendering zeros in outFields
                getTokenScorer(
                    fieldScorer,
                    text,
                    ScoreType.EXACT);
            } else if (phraseQuery(query)) {
                final ScoreType scoreType =
                    quoted ? ScoreType.EXACT : ScoreType.NON_EXACT;
                final TokenScorerFactory scorer =
                    getTokenScorer(
                        fieldScorer,
                        text,
                        scoreType);
                query = new FieldScoreQueryFilter(field, query, scorer);
            }
            return query;
        }

        private boolean phraseQuery(final Query query) {
            return query instanceof PhraseQuery
                || query instanceof MultiPhraseQuery;
        }

        private Query slopPhrase(final Query query) {
            if (query instanceof PhraseQuery) {
                ((PhraseQuery) query).setSlop(DEFAULT_AUTO_SLOP);
            } else if (query instanceof MultiPhraseQuery) {
                ((MultiPhraseQuery) query).setSlop(DEFAULT_AUTO_SLOP);
            }
            return query;
        }

        @Override
        protected Query getFieldQueryBase(final String field, final String text,
            final boolean quoted)
            throws ParseException
        {
            Query query = super.getFieldQueryBase(field, text, false);
            if (phraseQuery(query)) {
                if (!quoted && getAutoGeneratePhraseQueries()) {
                    //add to per field boolean query to autogenerated phrase query
                    setDefaultOperator(QueryParser.AND_OPERATOR);
                    BooleanQuery merged = new BooleanQuery(true);
                    merged.add(
                        wrapQuery(
                            query,
                            field,
                            text,
                            true),
                            BooleanClause.Occur.SHOULD);
                    //PhraseQueries has no clone() so parse again
                    Query slopped = super.getFieldQueryBase(field, text, false);
                    merged.add(
                        wrapQuery(
                            slopPhrase(slopped),
                            field,
                            text,
                            false), BooleanClause.Occur.SHOULD);
                    query = merged;
                }
            } else {
                query = wrapQuery(
                    query,
                    field,
                    text,
                    quoted);
            }
            return query;
        }

        @Override
        protected Query newQuotedQuery(final String field, final String text)
            throws ParseException
        {
            //register empty scorer for rendering zeros in outFields
            final FieldScorerFactory fieldScorer = perFieldScorers.get(field);
            if (fieldScorer == null) {
                return super.newQuotedQuery(field, text);
            }
            getTokenScorer(
                fieldScorer,
                text,
                ScoreType.NON_EXACT);
            return wrapQuery(
                super.newQuotedQuery(field, text),
                field,
                text,
                true);
        }

        @Override
        protected Query newPrefixQuery(final Term prefix) {
            return wrapQuery(
                super.newPrefixQuery(prefix),
                prefix.field(),
                prefix.text(),
                false);
        }

        @Override
        protected Query newWildcardQuery(final Term prefix) {
            return wrapQuery(
                super.newWildcardQuery(prefix),
                prefix.field(),
                prefix.text(),
                false);
        }
    }
}
