package ru.yandex.ps.search;

import java.io.File;
import java.io.FileWriter;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;

import ru.yandex.msearch.Config;
import ru.yandex.msearch.DatabaseManager;
import ru.yandex.msearch.FieldConfig;
import ru.yandex.msearch.config.DatabaseConfig;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.ps.search.field.GlobalField;
import ru.yandex.ps.search.field.GlobalStoredField;
import ru.yandex.ps.search.field.PrefixedField;
import ru.yandex.ps.search.field.PrefixedStoredField;
import ru.yandex.ps.search.field.impl.BasicSearchBackendField;
import ru.yandex.ps.search.field.impl.GlobalFieldImpl;
import ru.yandex.ps.search.field.impl.GlobalStoredFieldImpl;
import ru.yandex.ps.search.field.impl.PrefixedFieldImpl;
import ru.yandex.ps.search.field.impl.PrefixedStoredFieldImpl;

public class SearchBackendDaoFieldsGenerator {
    public static void main(String[] args) throws Exception {
        String packageName = args[0];
        String output = args[1];
        for (int i = 2; i < args.length; i+=2) {
            generate(args[i], args[i+1], packageName, output);
        }
    }

    private static void generate(
        final String configPath,
        final String scope,
        final String packageName,
        final String outputPath)
        throws Exception
    {
        System.err.println(
            "Generating, config " + configPath + " scope "
                + scope + " package " + packageName + " output " + outputPath);

        IniConfig config = new IniConfig(new File(configPath));
        Map<String, DatabaseConfig> configs = Config.parseDatabases(config);
        if (configs.size() == 0) {
            throw new Exception("No databases/fields found " + configPath);
        }

        if (configs.size() == 1) {
            generate(configs.entrySet().iterator().next().getValue(), scope, packageName, outputPath);
        } else {
            for (Map.Entry<String, DatabaseConfig> entry: configs.entrySet()) {
                if (DatabaseManager.DEFAULT_DATABASE.equalsIgnoreCase(entry.getKey())) {
                    generate(entry.getValue(), scope, packageName, outputPath);
                } else {
                    generate(entry.getValue(), scope + capitalize(entry.getKey()), packageName, outputPath);
                }
            }
        }
    }

    private static void generate(
        final DatabaseConfig dbConfig,
        final String scope,
        final String packageName,
        final String outputPath)
        throws Exception
    {
        System.err.println(
            "Generating, database " + dbConfig.name() + " scope "
                + scope + " package " + packageName + " output " + outputPath);

        String imports =
            "import ru.yandex.ps.search.field.GlobalField;\n" +
                "import ru.yandex.ps.search.field.GlobalStoredField;\n" +
                "import ru.yandex.ps.search.field.PrefixedField;\n" +
                "import ru.yandex.ps.search.field.PrefixedStoredField;\n" +
                "import ru.yandex.ps.search.field.impl.BasicSearchBackendField;\n" +
                "import ru.yandex.ps.search.field.impl.GlobalFieldImpl;\n" +
                "import ru.yandex.ps.search.field.impl.GlobalStoredFieldImpl;\n" +
                "import ru.yandex.ps.search.field.impl.PrefixedFieldImpl;\n" +
                "import ru.yandex.ps.search.field.impl.PrefixedStoredFieldImpl;\n";


        String className = capitalize(scope) + "Fields";
        String output = outputPath + "/" + className + ".java";

        StringBuilder classBody = new StringBuilder("package ");
        classBody.append(packageName);
        classBody.append(";\n\n");
        classBody.append(imports);
        classBody.append("\npublic class ");
        classBody.append(className);
        classBody.append(" {\n");
        String namePrefix = scope + '_';
        BackendFieldsConfig config = new BackendFieldsConfig(dbConfig);
        Set<String> generated = new LinkedHashSet<>();
        for (Map.Entry<String, List<FieldConfig>> entry
            : config.fieldsWithAliases().entrySet())
        {
            String name = entry.getKey();
            if (generated.contains(name)) {
                continue;
            }

            List<FieldConfig> aliases = entry.getValue();
            String fieldName = name;
            if (name.startsWith(namePrefix)) {
                fieldName = name.substring(namePrefix.length());
            }

            classBody.append("    public static final ");
            if (aliases.size() == 1) {
                handleNonAliased(classBody, fieldName, aliases.get(0));
                generated.add(name);
                continue;
            }

            if (aliases.size() == 2 ) {
                if (handleTwoAliases(classBody, fieldName, aliases.get(0), aliases.get(1))) {
                    generated.add(aliases.get(0).field());
                    generated.add(aliases.get(1).field());
                    continue;
                }
            }

            handleNonAliased(classBody, fieldName, aliases.get(0));
            generated.add(aliases.get(0).field());
        }

        classBody.append("}");

        try (FileWriter writer = new FileWriter(new File(output))) {
            writer.write(classBody.toString());
        }

        System.err.println(output);
        System.err.println(classBody);
    }

    private static void handlePrefixAliases(
        final StringBuilder classBody,
        final String fieldName,
        final FieldConfig prefixed,
        final FieldConfig nonPrefixed)
        throws Exception
    {
        if (prefixed.store() && nonPrefixed.store()) {
            throw new Exception("Only one from alias chain should be stored");
        }

        String stored = null;
        if (prefixed.store()) {
            stored = "\"" + prefixed.field() + "\"";
        } else if (nonPrefixed.store()) {
            stored = "\"" + nonPrefixed.field() + "\"";
        }

        classBody.append(BasicSearchBackendField.class.getSimpleName());
        classBody.append(" ");
        classBody.append(variableName(fieldName));
        classBody.append(" = new ");
        classBody.append(BasicSearchBackendField.class.getSimpleName());
        classBody.append("(\"");
        classBody.append(fieldName);
        classBody.append("\",");
        classBody.append(stored);
        classBody.append(",\"");
        classBody.append(prefixed.field());
        classBody.append("\",\"");
        classBody.append(nonPrefixed.field());
        classBody.append("\");\n");
    }

    private static boolean handleTwoAliases(
        final StringBuilder classBody,
        final String fieldName,
        final FieldConfig first,
        final FieldConfig second)
        throws Exception
    {
        if (first.prefixed() && !second.prefixed()) {
            handlePrefixAliases(classBody, fieldName, first, second);
        } else if (!first.prefixed() && second.prefixed()) {
            handlePrefixAliases(classBody, fieldName, second, first);
        } else {
            return false;
        }

        return true;
    }

    private static void handleNonAliased(
        final StringBuilder classBody,
        final String fieldName,
        final FieldConfig fieldConfig)
        throws Exception
    {
        if (fieldConfig.store()) {
            if (fieldConfig.prefixed() && fieldConfig.store()) {
                classBody.append(PrefixedStoredField.class.getSimpleName());
                classBody.append(" ");
                classBody.append(variableName(fieldName));
                classBody.append(" = new ");
                classBody.append(PrefixedStoredFieldImpl.class.getSimpleName());
                classBody.append("(\"");
                classBody.append(fieldConfig.field());
                classBody.append("\",\"");
                classBody.append(fieldConfig.field());
                classBody.append("\",\"");
                classBody.append(fieldConfig.field());
                classBody.append("\");\n");
            } else if (!fieldConfig.prefixed() && fieldConfig.store()) {
                classBody.append(GlobalStoredField.class.getSimpleName());
                classBody.append(" ");
                classBody.append(variableName(fieldName));
                classBody.append(" = new ");
                classBody.append(GlobalStoredFieldImpl.class.getSimpleName());
                classBody.append("(\"");
                classBody.append(fieldConfig.field());
                classBody.append("\",\"");
                classBody.append(fieldConfig.field());
                classBody.append("\",\"");
                classBody.append(fieldConfig.field());
                classBody.append("\");\n");
            }
        } else {
            if (fieldConfig.prefixed()) {
                classBody.append(PrefixedField.class.getSimpleName());
                classBody.append(" ");
                classBody.append(variableName(fieldName));
                classBody.append(" = new ");
                classBody.append(PrefixedFieldImpl.class.getSimpleName());
                classBody.append("(\"");
                classBody.append(fieldConfig.field());
                classBody.append("\",\"");
                classBody.append(fieldConfig.field());
                classBody.append("\");\n");
            } else {
                classBody.append(GlobalField.class.getSimpleName());
                classBody.append(" ");
                classBody.append(variableName(fieldName));
                classBody.append(" = new ");
                classBody.append(GlobalFieldImpl.class.getSimpleName());
                classBody.append("(\"");
                classBody.append(fieldConfig.field());
                classBody.append("\",\"");
                classBody.append(fieldConfig.field());
                classBody.append("\");\n");
            }
        }
    }
    private static String capitalize(String s) {
        return s.substring(0,1).toUpperCase() + s.substring(1);
    }

    private static String variableName(final String name) {
        String result = name.toUpperCase(Locale.ENGLISH);
        if (!Character.isLetter(result.charAt(0))) {
            result = "F_" + result;
        }
        return result;
    }
}
