package ru.yandex.search.json;

import java.util.ArrayDeque;
import java.util.Deque;

import ru.yandex.json.parser.ContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.StringCollector;
import ru.yandex.json.parser.StringCollectorFactory;

public class HandlersManager implements ContentHandler {
    private static final String EXCEPTION_AT = "Exception at ";

    private final Deque<ContentHandler> handlers = new ArrayDeque<>();
    private final StringCollector sc;
    private ContentHandler current = null;

    public HandlersManager() {
        this(StringCollectorFactory.INSTANCE.create());
    }

    public HandlersManager(final StringCollector sc) {
        this.sc = sc;
    }

    public StringCollector sc() {
        return sc;
    }

    public void push(final ContentHandler handler) {
        handlers.push(handler);
        current = handler;
    }

    public void pop() {
        handlers.pop();
        current = handlers.peek();
    }

    private void handleException(final JsonException e) throws JsonException {
        // Check if we won't change the exception class while wrapping
        if (e.getClass().equals(JsonException.class)) {
            throw new JsonException(EXCEPTION_AT + handlers, e);
        } else {
            throw e;
        }
    }

    private void handleException(final JsonException e, final Object value)
        throws JsonException
    {
        if (e.getClass().equals(JsonException.class)) {
            throw new JsonException(
                EXCEPTION_AT + handlers + " while processing " + value,
                e);
        } else {
            throw e;
        }
    }

    @Override
    public void startObject() throws JsonException {
        try {
            current.startObject();
        } catch (JsonException e) {
            handleException(e);
        }
    }

    @Override
    public void endObject() throws JsonException {
        try {
            current.endObject();
        } catch (JsonException e) {
            handleException(e);
        }
    }

    @Override
    public void startArray() throws JsonException {
        try {
            current.startArray();
        } catch (JsonException e) {
            handleException(e);
        }
    }

    @Override
    public void endArray() throws JsonException {
        try {
            current.endArray();
        } catch (JsonException e) {
            handleException(e);
        }
    }

    // CSOFF: ParameterNumber
    @Override
    public void key(
        final char[] buf,
        final int off,
        final int len,
        final boolean eol)
        throws JsonException
    {
        try {
            current.key(buf, off, len, eol);
        } catch (JsonException e) {
            handleException(e, new String(buf, off, len));
        }
    }

    @Override
    public void value(
        final char[] buf,
        final int off,
        final int len,
        final boolean eol)
        throws JsonException
    {
        try {
            current.value(buf, off, len, eol);
        } catch (JsonException e) {
            handleException(e, new String(buf, off, len));
        }
    }
    // CSON: ParameterNumber

    @Override
    public void value(final long value) throws JsonException {
        try {
            current.value(value);
        } catch (JsonException e) {
            handleException(e, value);
        }
    }

    @Override
    public void value(final double value) throws JsonException {
        try {
            current.value(value);
        } catch (JsonException e) {
            handleException(e, value);
        }
    }

    @Override
    public void value(final boolean value) throws JsonException {
        try {
            current.value(value);
        } catch (JsonException e) {
            handleException(e, value);
        }
    }

    @Override
    public void nullValue() throws JsonException {
        try {
            current.nullValue();
        } catch (JsonException e) {
            handleException(e, null);
        }
    }

    @Override
    public void endObjectEntry() throws JsonException {
        try {
            current.endObjectEntry();
        } catch (JsonException e) {
            handleException(e);
        }
    }
}

