package ru.yandex.search.json.fieldfunction;

import java.util.List;
import java.util.Map;

import ru.yandex.search.json.fieldfunction.value.FunctionValue;
import ru.yandex.search.json.fieldfunction.value.IntValue;

public class IncrementFunction implements FieldFunction, FunctionConstructor {
    private static final long DEFAULT_INCREMENT = 1;

    private final FieldFunction increment;
    private final FieldFunction condition;
    private final String fieldName;

    public IncrementFunction() {
        condition = null;
        increment = null;
        fieldName = null;
    }

    public IncrementFunction(
        final String fieldName,
        final FieldFunction increment)
    {
        this.fieldName = fieldName;
        this.increment = increment;
        this.condition = null;
    }

    public IncrementFunction(
        final String fieldName,
        final FieldFunction increment,
        final FieldFunction condition)
    {
        this.fieldName = fieldName;
        this.increment = increment;
        this.condition = condition;
    }

    @Override
    public final FunctionValue value(
        final FieldAccessor accessor,
        final ConditionsAccessor condAccessor)
        throws FieldFunctionException
    {
        FunctionValue fieldValue = accessor.getFieldValue(fieldName);
        if (condition != null) {
            FunctionValue condValue = condition.value(accessor, condAccessor);
            if (!condValue.booleanValue()) {
                //noop
                return fieldValue;
            }
        }
        final FunctionValue ret;
        if (increment != null) {
            ret = new IntValue(
                fieldValue.intValue()
                + increment.value(accessor, condAccessor).intValue());
        } else {
            ret = new IntValue(fieldValue.intValue() + DEFAULT_INCREMENT);
        }
        return ret;
    }

    @Override
    public final FieldFunction construct(
        final String fieldName,
        final List<FieldFunction> args,
        final Map<String, String> conditions)
        throws FieldFunctionException
    {
        final int argsSize = args.size();
        final FieldFunction ret;
        switch (argsSize) {
            case 0:
                ret = new IncrementFunction(fieldName, null);
                break;
            case 1:
                ret = new IncrementFunction(fieldName, args.get(0));
                break;
            case 2:
                ret = new IncrementFunction(
                    fieldName,
                    args.get(0),
                    args.get(1));
                break;
            default:
                throw new FieldFunctionException(
                    "IncrementFunction: invalid argument count <"
                    + argsSize + ">. Usage: inc([step],[condition])");
        }
        return ret;
    }

    @Override
    public String toString() {
        StringBuilder desc = new StringBuilder("IncrementFunction( ");
        if (increment != null) {
            desc.append(increment);
        } else {
            desc.append(DEFAULT_INCREMENT);
        }
        if (condition != null) {
            desc.append(", ");
            desc.append(condition);
        }
        desc.append(')');
        return desc.toString();
    }
}
