package ru.yandex.search.json.fieldfunction;

import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import ru.yandex.search.json.fieldfunction.value.FunctionValue;
import ru.yandex.search.json.fieldfunction.value.StringValue;
import ru.yandex.util.string.StringUtils;

public class MakeSetFunction implements FieldFunction, FunctionConstructor {
    public static final String MAKE_SET = "make_set";

    private static final FieldFunction DEFAULT_MAX_NAMES =
        new ConstantValueFieldFunction(Long.MAX_VALUE);

    private final FieldFunction arg1;
    private final FieldFunction arg2;
    private final FieldFunction maxNames;

    public MakeSetFunction() {
        arg1 = null;
        arg2 = null;
        maxNames = null;
    }

    public MakeSetFunction(
        final FieldFunction arg1,
        final FieldFunction arg2,
        final FieldFunction maxNames)
    {
        this.arg1 = arg1;
        this.arg2 = arg2;
        this.maxNames = maxNames;
    }

    @Override
    public final FunctionValue value(
        final FieldAccessor accessor,
        final ConditionsAccessor condAccessor)
        throws FieldFunctionException
    {
        return new StringValue(
            makeSet(
                arg1.value(accessor, condAccessor).stringValue(),
                arg2.value(accessor, condAccessor).stringValue(),
                maxNames.value(accessor, condAccessor).intValue()));
    }

    @Override
    public final FieldFunction construct(
        final String fieldName,
        final List<FieldFunction> args,
        final Map<String, String> conditions)
        throws FieldFunctionException
    {
        FieldFunction maxNames;
        switch (args.size()) {
            case 2:
                maxNames = DEFAULT_MAX_NAMES;
                break;
            case 2 + 1:
                maxNames = args.get(2);
                break;
            default:
                throw new FieldFunctionException(
                    "MakeSetFunction: invalid argument count <"
                    + args.size() + ">, but should be 2 or 3");
        }
        return new MakeSetFunction(args.get(0), args.get(1), maxNames);
    }

    @Override
    public String toString() {
        return "MakeSetFunction(" + arg1
            + ',' + ' ' + arg2
            + ',' + ' ' + maxNames + ')';
    }

    private static String makeSet(
        final String ones,
        final String others,
        final long maxNames)
    {
        int idx = ones.indexOf(others);
        if (idx != -1
            && (idx == 0 || ones.charAt(idx - 1) == '\n')
            && (idx + others.length() >= ones.length()
                || ones.charAt(idx + others.length()) == '\n'))
        {
            return ones;
        }

        Set<String> set = new LinkedHashSet<>();
        addToSet(set, others, maxNames);
        addToSet(set, ones, maxNames);
        return StringUtils.join(set, '\n');
    }

    @SuppressWarnings("StringSplitter")
    public static void addToSet(
        final Set<String> set,
        final String strings,
        final long maxNames)
    {
        if (set.size() < maxNames) {
            for (String str: strings.split("\n")) {
                String trim = str.trim();
                if (!trim.isEmpty()
                    && set.add(trim)
                    && set.size() == maxNames)
                {
                    break;
                }
            }
        }
    }
}

