package ru.yandex.search.json.fieldfunction;

import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import ru.yandex.search.json.fieldfunction.value.FunctionValue;
import ru.yandex.search.json.fieldfunction.value.StringValue;

public class SumMapFunction implements FieldFunction, FunctionConstructor {
    public static final String KV_SEPARATOR = "\t";
    public static final String RECORD_SEPARATOR = "\n";
    public static final String EMPTY_VALUE = "0";

    private final FieldFunction arg1;
    private final FieldFunction arg2;
    private final boolean keepOrder;

    public SumMapFunction() {
        this.arg1 = null;
        this.arg2 = null;
        this.keepOrder = false;
    }

    public SumMapFunction(final FieldFunction arg1, final FieldFunction arg2) {
        this.arg1 = arg1;
        this.arg2 = arg2;
        this.keepOrder = false;
    }

    public SumMapFunction(final FieldFunction arg1, final FieldFunction arg2, final boolean keepOrder) {
        this.arg1 = arg1;
        this.arg2 = arg2;
        this.keepOrder = keepOrder;

    }

    @Override
    public FunctionValue value(
        final FieldAccessor accessor,
        final ConditionsAccessor condAccessor)
        throws FieldFunctionException
    {
        return new StringValue(
            updateMap(
                arg1.value(accessor, condAccessor).stringValue(),
                arg2.value(accessor, condAccessor).stringValue()));
    }

    protected String updateMap(final String first, final String second)
        throws FieldFunctionException
    {
        Map<String, Long> map;
        if (keepOrder) {
            map = new LinkedHashMap<>();
        } else {
            map = new HashMap<>();
        }
        addToMap(map, first);
        addToMap(map, second);
        return mapToString(map);
    }

    public static String mapToString(final Map<String, Long> map) {
        StringBuilder sb = new StringBuilder();
        for (Map.Entry<String, Long> entry: map.entrySet()) {
            sb.append(entry.getKey());
            sb.append(KV_SEPARATOR);
            sb.append(entry.getValue());
            sb.append(RECORD_SEPARATOR);
        }

        if (sb.length() > 0) {
            sb.setLength(sb.length() - 1);
        }

        return sb.toString();
    }

    @SuppressWarnings("StringSplitter")
    public static void parseMap(
        final Map<String, Long> map,
        final String stringValue)
    {
        for (String line: stringValue.split(RECORD_SEPARATOR)) {
            String trimLine = line.trim();
            if (trimLine.isEmpty()) {
                continue;
            }

            String key;
            Long value;
            String[] kv = line.trim().split(KV_SEPARATOR);
            if (kv.length != 2) {
                if (kv.length == 1) {
                    //convertation from make_set support
                    key = kv[0].trim();
                    value = 1L;
                } else {
                    throw new NumberFormatException(
                        "SumMapFunciton: wrong format " + stringValue);
                }
            } else {
                key = kv[0].trim();
                value = Long.parseLong(kv[1]);
            }

            Long currentValue = map.get(key);

            if (currentValue != null) {
                value += currentValue;
            }

            if (value != 0) {
                map.put(key, value);
            } else {
                map.remove(key);
            }
        }
    }

    public static void addToMap(
        final Map<String, Long> map,
        final String str)
        throws FieldFunctionException
    {
        String trimLine = str.trim();
        if (trimLine.equals(EMPTY_VALUE) || trimLine.isEmpty()) {
            return;
        }

        try {
            parseMap(map, trimLine);
        } catch (NumberFormatException nfe) {
            throw new FieldFunctionException(
                "SumMapFunction wrong format, value not integer " + str);
        }
    }

    @Override
    public FieldFunction construct(
        final String fieldName,
        final List<FieldFunction> args,
        final Map<String, String> conditions)
        throws FieldFunctionException
    {
        if (args.size() == 3) {
            return new SumMapFunction(
                args.get(0),
                args.get(1),
                args.get(2).value(null, null).booleanValue());
        }

        if (args.size() != 2) {
            throw new FieldFunctionException(
                "SumMapFunction: invalid argument count <"
                    + args.size() + ">, but should be 2");
        }

        return new SumMapFunction(args.get(0), args.get(1));
    }
}
