package ru.yandex.search.proxy;

import java.util.Map;

import ru.yandex.collection.Pattern;
import ru.yandex.collection.PatternMap;
import ru.yandex.function.GenericFunction;
import ru.yandex.http.util.request.RequestInfo;
import ru.yandex.http.util.request.RequestPatternParser;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;

public abstract class AbstractUpstreamsConfigBuilder
    <T extends AbstractUpstreamsConfigBuilder<T>>
    implements UpstreamsConfig
{
    public static final String SECTION = "upstream";

    private static final GenericFunction<
        UpstreamConfig,
        UpstreamConfigBuilder,
        RuntimeException> TRANSFORMER = new GenericFunction<
            UpstreamConfig,
            UpstreamConfigBuilder,
            RuntimeException>()
            {
                @Override
                public UpstreamConfigBuilder apply(
                    final UpstreamConfig config)
                {
                    if (config == null) {
                        return null;
                    } else {
                        return new UpstreamConfigBuilder(config);
                    }
                }
            };

    private PatternMap<RequestInfo, UpstreamConfigBuilder> upstreams;

    protected AbstractUpstreamsConfigBuilder() {
        this(UpstreamsConfigDefaults.INSTANCE);
    }

    protected AbstractUpstreamsConfigBuilder(final UpstreamsConfig config) {
        upstreams = config.upstreams().transform(TRANSFORMER);
    }

    protected AbstractUpstreamsConfigBuilder(final IniConfig config)
        throws ConfigException
    {
        this(config, UpstreamsConfigDefaults.INSTANCE);
    }

    protected AbstractUpstreamsConfigBuilder(
        final IniConfig config,
        final UpstreamsConfig defaults)
        throws ConfigException
    {
        upstreams = loadUpstreams(
            config,
            defaults);
    }

    public static PatternMap<RequestInfo, UpstreamConfigBuilder>
        loadUpstreams(
            final IniConfig section,
            final UpstreamsConfig defaults)
            throws ConfigException
    {
        PatternMap<RequestInfo, UpstreamConfigBuilder> upstreams =
            new PatternMap<>(
                new UpstreamConfigBuilder(
                    section,
                    defaults.upstreams().asterisk()));

        for (Map.Entry<String, IniConfig> entry
                : section.sections().entrySet())
        {
            String subsection = entry.getKey();
            if (subsection.charAt(0) == '/') {
                Pattern<RequestInfo> pattern =
                    RequestPatternParser.INSTANCE.apply(subsection);
                upstreams.put(
                    pattern,
                    new UpstreamConfigBuilder(
                        entry.getValue(),
                        defaults.upstreams().get(pattern)));
            }
        }
        return upstreams;
    }

    @Override
    public PatternMap<RequestInfo, UpstreamConfigBuilder> upstreams() {
        return upstreams;
    }

    public T upstreams(
        final PatternMap<RequestInfo, ? extends UpstreamConfig> upstreams)
    {
        this.upstreams = upstreams.transform(TRANSFORMER);
        return self();
    }

    public T upstream(
        final Pattern<RequestInfo> pattern,
        final UpstreamConfigBuilder upstream)
    {
        upstreams.put(pattern, upstream);
        return self();
    }

    public T asterisk(final UpstreamConfigBuilder upstream) {
        upstreams.put(new Pattern<>("", true), upstream);
        return self();
    }

    public ImmutableUpstreamsConfig build() throws ConfigException {
        return new ImmutableUpstreamsConfig(this);
    }

    protected abstract T self();
}

