package ru.yandex.search.proxy;

import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.function.Supplier;

import org.apache.http.HttpException;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpHost;
import org.apache.http.client.protocol.HttpClientContext;

import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.NotFoundException;
import ru.yandex.http.util.PayloadFutureCallback;
import ru.yandex.http.util.ServiceUnavailableException;
import ru.yandex.http.util.StackTraceFutureCallback;
import ru.yandex.http.util.nio.AsyncStringConsumerFactory;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.request.RequestInfo;
import ru.yandex.http.util.server.HttpServer;
import ru.yandex.parser.searchmap.User;

public class BroadcastHandler implements ProxyRequestHandler {
    private final SearchProxy<? extends ImmutableSearchProxyConfig> proxy;

    public BroadcastHandler(
        final SearchProxy<? extends ImmutableSearchProxyConfig> proxy)
    {
        this.proxy = proxy;
    }

    @Override
    public void handle(final ProxySession session) throws HttpException {
        UpstreamContext upstream =
            proxy.upstreamFor(
                (RequestInfo) session.context().getAttribute(
                    HttpServer.REQUEST_INFO));
        String path =
            upstream.removePrefix(session.request().getRequestLine().getUri());
        User user = proxy.extractUser(session);
        BasicAsyncRequestProducerGenerator generator;
        try {
            generator = new BasicAsyncRequestProducerGenerator(
                path,
                session.request());
        } catch (IOException e) {
            throw new ServiceUnavailableException(e);
        }
        generator.copyHeader(session.request(), HttpHeaders.ACCEPT_CHARSET);
        generator.copyHeader(session.request(), HttpHeaders.ACCEPT_ENCODING);
        List<HttpHost> hosts;
        if (session.params().getBoolean("index", false)) {
            hosts = proxy.searchMap().indexerHosts(user);
        } else {
            hosts = proxy.searchMap().searchHosts(user);
        }
        hosts = SearchProxy.shuffle(hosts, user);
        if (hosts.isEmpty()) {
            throw new NotFoundException("No hosts found for " + user);
        }

        AsyncClient client = upstream.client().adjust(session.context());
        Supplier<? extends HttpClientContext> contextGenerator =
            session.listener().createContextGeneratorFor(client);
        MultiFutureCallback<Map.Entry<HttpHost, String>> callback =
            new MultiFutureCallback<>(new BroadcastCallback(session));
        for (HttpHost host: hosts) {
            session.subscribeForCancellation(
                client.execute(
                    host,
                    generator,
                    AsyncStringConsumerFactory.INSTANCE,
                    contextGenerator,
                    new StackTraceFutureCallback(
                        new PayloadFutureCallback<>(
                            host,
                            callback.newCallback()))));
        }
        callback.done();
    }

    @Override
    public String toString() {
        return "Forwards request to all servers containing user index";
    }
}

