package ru.yandex.search.proxy;

import java.util.Arrays;
import java.util.function.LongPredicate;

public class Dataset {
    private static final int PERC = 100;

    private final long[] dataset;
    private final double mean;
    private final long max;

    public Dataset(final long[] dataset, final LongPredicate dataFilter) {
        long[] tmp = new long[dataset.length];
        int size = 0;
        long sum = 0L;
        for (long point: dataset) {
            if (dataFilter.test(point)) {
                tmp[size++] = point;
                sum += point;
            }
        }
        this.dataset = Arrays.copyOf(tmp, size);
        Arrays.sort(this.dataset);
        if (size == 0) {
            mean = 0d;
            max = 0L;
        } else {
            mean = ((double) sum) / size;
            max = this.dataset[size - 1];
        }
    }

    public int size() {
        return dataset.length;
    }

    public double mean() {
        return mean;
    }

    public long max() {
        return max;
    }

    public long[] percentiles(final int... percentiles) {
        long[] values = new long[percentiles.length];
        if (dataset.length != 0) {
            for (int i = 0; i < percentiles.length; ++i) {
                values[i] =
                    dataset[
                        Math.min(
                            percentiles[i] * dataset.length / PERC,
                            dataset.length - 1)];
            }
        }
        return values;
    }

    private void accountPoint(
        final long point,
        final int[] matches,
        final long[] boundaries)
    {
        for (int i = 0; i < boundaries.length; ++i) {
            if (point <= boundaries[i]) {
                ++matches[i];
                return;
            }
        }
    }

    // boundaries expected to be non-empty and sorted in ascending order
    public double[] dissection(final long... boundaries) {
        double[] dissection = new double[boundaries.length];
        if (dataset.length == 0) {
            Arrays.fill(dissection, PERC);
        } else {
            int[] matches = new int[boundaries.length];
            for (long point: dataset) {
                accountPoint(point, matches, boundaries);
            }
            double total = 0d;
            for (int i = 0; i < matches.length; ++i) {
                total += matches[i];
                dissection[i] = total * PERC / dataset.length;
            }
        }
        return dissection;
    }
}

