package ru.yandex.search.proxy;

import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.charset.CharsetEncoder;
import java.nio.charset.CodingErrorAction;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Supplier;

import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.HttpStatus;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.entity.ContentType;

import ru.yandex.charset.StreamEncoder;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.ErrorSuppressingFutureCallback;
import ru.yandex.http.util.FixedMultiFutureCallback;
import ru.yandex.http.util.RequestErrorType;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.NByteArrayEntityFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.io.DecodableByteArrayOutputStream;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonNull;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.parser.searchmap.SearchMap;
import ru.yandex.parser.string.PositiveLongValidator;

public class LagsStatHandler implements ProxyRequestHandler {
    private static final long DEFAULT_TTL = 600000L;
    private static final int MAX_UPSTREAMS_TO_LOG = 14;

    private static final long[] TIME_DISSECTION = new long[] {
        50L,
        150L,
        300L,
        500L,
        1000L,
        3000L,
        10000L,
        30000L,
        60000L,
        300000L,
        600000L,
        3600000L,
        86400000L
    };

    private static final int[] PERCENTILES =
        new int[] {50, 75, 90, 95, 98, 99};

    private final SearchProxy<? extends ImmutableSearchProxyConfig> proxy;
    private final AsyncClient client;

    public LagsStatHandler(
        final SearchProxy<? extends ImmutableSearchProxyConfig> proxy,
        final AsyncClient client)
    {
        this.proxy = proxy;
        this.client = client;
    }

    @Override
    public void handle(final ProxySession session) throws HttpException {
        String uri =
            "/lags?json-type=dollar&expired-value=" + Long.MIN_VALUE + "&ttl="
            + session.params().get(
                "ttl",
                DEFAULT_TTL,
                PositiveLongValidator.INSTANCE);
        BasicAsyncRequestProducerGenerator producerGenerator =
            new BasicAsyncRequestProducerGenerator(uri);

        List<HttpHost> hosts =
            new ArrayList<>(proxy.searchMap().indexerHosts());
        int size = hosts.size();
        FixedMultiFutureCallback<JsonObject> callback =
            new FixedMultiFutureCallback<>(
                new Callback(
                    session,
                    JsonTypeExtractor.NORMAL.extract(session.params())),
                size);
        AsyncClient client;
        Supplier<? extends HttpClientContext> contextGenerator;
        int maxUpstreamsToLog = session.params().getInt(
            "max-upstreams-to-log",
            MAX_UPSTREAMS_TO_LOG);
        if (size <= maxUpstreamsToLog) {
            client = this.client.adjust(session.context());
            contextGenerator =
                session.listener().createContextGeneratorFor(client);
        } else {
            client = this.client;
            contextGenerator = client.httpClientContextGenerator();
        }
        for (int i = 0; i < size; ++i) {
            session.subscribeForCancellation(
                client.execute(
                    hosts.get(i),
                    producerGenerator,
                    JsonAsyncTypesafeDomConsumerFactory.OK,
                    contextGenerator,
                    new ErrorSuppressingFutureCallback<>(
                        callback.callback(i),
                        x -> RequestErrorType.HTTP,
                        JsonNull.INSTANCE)));
        }
    }

    @Override
    public String toString() {
        return "Indexing lag stats";
    }

    private static class Callback
        extends AbstractProxySessionCallback<List<JsonObject>>
    {
        private final long start = System.currentTimeMillis();
        private final JsonType jsonType;

        Callback(final ProxySession session, final JsonType jsonType) {
            super(session);
            this.jsonType = jsonType;
        }

        private static void writeValue(
            final JsonWriter writer,
            final String name,
            final Number value)
            throws IOException
        {
            writer.startArray();
            writer.value(name + "_avvv");
            writer.value(value);
            writer.endArray();
            writer.startArray();
            writer.value(name + "_axxx");
            writer.value(value);
            writer.endArray();
        }

        @Override
        public void completed(final List<JsonObject> results) {
            long end = System.currentTimeMillis();
            try {
                Map<String, long[]> lags = new HashMap<>();
                int aliveBackends = 0;
                for (JsonObject result: results) {
                    if (result != JsonNull.INSTANCE) {
                        ++aliveBackends;
                        for (Map.Entry<String, JsonObject> entry
                            : result.asMap().entrySet())
                        {
                            String queue = entry.getKey();
                            long[] queueLags = lags.get(queue);
                            if (queueLags == null) {
                                queueLags =
                                    new long[(int) SearchMap.SHARDS_COUNT];
                                Arrays.fill(queueLags, Long.MAX_VALUE);
                                lags.put(queue, queueLags);
                            }
                            for (Map.Entry<String, JsonObject> queueEntry
                                : entry.getValue().asMap().entrySet())
                            {
                                String shardString = queueEntry.getKey();
                                int shard;
                                try {
                                    shard = Integer.parseInt(shardString);
                                } catch (NumberFormatException e) {
                                    throw new JsonException(
                                        "Failed to parse shard number for "
                                        + "queue " + queue
                                        + " from string " + shardString,
                                        e);
                                }
                                long lag = queueEntry.getValue().asLong();
                                long minLag = queueLags[shard];
                                if (lag < minLag) {
                                    queueLags[shard] = lag;
                                }
                            }
                        }
                    }
                }
                session.logger().info(
                    aliveBackends + " backends responded in "
                    + (end - start) + " ms. Responses processed in "
                    + (System.currentTimeMillis() - end) + " ms");
                Charset charset = session.acceptedCharset();
                CharsetEncoder encoder = charset.newEncoder()
                    .onMalformedInput(CodingErrorAction.REPLACE)
                    .onUnmappableCharacter(CodingErrorAction.REPLACE);
                DecodableByteArrayOutputStream out =
                    new DecodableByteArrayOutputStream();
                try (JsonWriter writer =
                        jsonType.create(new StreamEncoder(out, encoder)))
                {
                    writer.startArray();
                    writer.startArray();
                    writer.value("indexation-lag-alive-backends_axxx");
                    writer.value(aliveBackends);
                    writer.endArray();
                    for (Map.Entry<String, long[]> entry: lags.entrySet()) {
                        String prefix = "indexation-lag-" + entry.getKey();
                        long[] data = entry.getValue();
                        Dataset dataset = new Dataset(
                            data,
                            x -> x != Long.MAX_VALUE && x != Long.MIN_VALUE);

                        writer.startArray();
                        writer.value(prefix + "-shards_axxx");
                        writer.value(dataset.size());
                        writer.endArray();

                        writeValue(writer, prefix + "-mean", dataset.mean());

                        long max = dataset.max();
                        writeValue(writer, prefix + "-max", max);
                        int shard = 0;
                        for (; shard < data.length; ++shard) {
                            if (data[shard] == max) {
                                break;
                            }
                        }

                        writer.startArray();
                        writer.value(prefix + "-worst-shard_axxx");
                        writer.value(shard);
                        writer.endArray();

                        long[] percentiles = dataset.percentiles(PERCENTILES);
                        for (int i = 0; i < PERCENTILES.length; ++i) {
                            writeValue(
                                writer,
                                prefix + '-' + PERCENTILES[i] + "prc",
                                percentiles[i]);
                        }

                        double[] dissection =
                            dataset.dissection(TIME_DISSECTION);
                        for (int i = 0; i < TIME_DISSECTION.length; ++i) {
                            writeValue(
                                writer,
                                prefix + '-' + TIME_DISSECTION[i] + "ms",
                                dissection[i]);
                        }
                    }
                    writer.endArray();
                }
                session.response(
                    HttpStatus.SC_OK,
                    out.processWith(
                        new NByteArrayEntityFactory(
                            ContentType.APPLICATION_JSON.withCharset(
                                charset))));
            } catch (Exception e) {
                failed(e);
                return;
            }
        }
    }
}

