package ru.yandex.search.proxy;

import java.io.IOException;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.text.ParseException;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.concurrent.ThreadLocalRandom;
import java.util.regex.Matcher;

import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.entity.ContentType;

import ru.yandex.collection.Pattern;
import ru.yandex.collection.PatternMap;
import ru.yandex.http.proxy.HttpProxy;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.server.async.DelegatedHttpAsyncRequestHandler;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.request.RequestHandlerMapper;
import ru.yandex.http.util.request.RequestInfo;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.parser.searchmap.SearchMap;
import ru.yandex.parser.searchmap.ShardNumberValidator;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.search.mop.manager.MopManager;
import ru.yandex.search.prefix.Prefix;
import ru.yandex.search.prefix.ShardPrefix;
import ru.yandex.stater.AlertThresholds;
import ru.yandex.stater.GolovanAlertsConfig;
import ru.yandex.stater.GolovanChart;
import ru.yandex.stater.GolovanChartGroup;
import ru.yandex.stater.GolovanPanel;
import ru.yandex.stater.GolovanSignal;
import ru.yandex.stater.ImmutableGolovanAlertsConfig;
import ru.yandex.stater.ImmutableGolovanPanelConfig;
import ru.yandex.stater.Stater;
import ru.yandex.stater.StatsConsumer;

public class SearchProxy<T extends ImmutableSearchProxyConfig>
    extends HttpProxy<T>
{
    private static final String SERVER_HOSTNAME = serverHostname();
    private static final String LOCALHOST = "localhost";
    private static final String SAS = "sas";
    private static final String MAN = "man";
    private static final String IVA = "iva";
    private static final String MYT = "myt";
    private static final String VLA = "vla";
    private static final java.util.regex.Pattern GENCFG_DC =
        java.util.regex.Pattern.compile(
            "^([a-z]{3}).*\\.gencfg-c\\.yandex\\.net$");
    private static final java.util.regex.Pattern YP_DC =
        java.util.regex.Pattern.compile(
            "^.*\\.([a-z]{3})\\.yp-c\\.yandex\\.net$");
    private static final String DC = dc(SERVER_HOSTNAME);
    private static final PingTable PING_TABLE = pingTable(DC);

    protected final AsyncClient searchClient;
    protected final AsyncClient indexerClient;
    protected final ContentType indexerContentType;
    protected final PatternMap<RequestInfo, UpstreamContext> upstreams;
    protected final MopManager mopManager;

    // use searchMap() method
    private final SearchMap searchMap;

    public SearchProxy(final T config) throws IOException {
        super(config);
        searchClient = client("Search", config.searchConfig());
        indexerClient = client("Indexer", config.indexerConfig());
        indexerContentType = ContentType.APPLICATION_JSON.withCharset(
            config.indexerConfig().requestCharset());
        upstreams = config.upstreamsConfig().prepareClients(this);
        if (config.searchMapConfig() != null) {
            try {
                searchMap = config.searchMapConfig().build();
            } catch (ParseException e) {
                throw new IOException("Failed to parse searchmap", e);
            }
            if (searchMap.reloadable()) {
                register(
                        new Pattern<>("/reloadsearchmap", false),
                        new DelegatedHttpAsyncRequestHandler<>(
                                new SearchMapReloader(searchMap),
                                this));
            }
        } else {
            searchMap = null;
        }
        if (config.mopManagerConfig() != null) {
            mopManager = new MopManager(
                config.mopManagerConfig(),
                config.dnsConfig(),
                logger());
        } else {
            mopManager = null;
        }
        if (config.searchMapConfig() == null && config.mopManagerConfig() == null) {
            throw new IOException("Both searchMapConfig and mopManagerConfig are null");
        }
        register(new Pattern<>("/unicast/", true), new UnicastHandler(this));
        register(
            new Pattern<>("/broadcast/", true),
            new BroadcastHandler(this));
        register(
            new Pattern<>("/info/prefix", false),
            new PrefixInfoHandler(searchMap()),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/lags-stat", false),
            new LagsStatHandler(this, indexerClient),
            RequestHandlerMapper.GET);
        registerStater(new LagsStatGolovanPanel());
    }

    private static String serverHostname() {
        String hostname = System.getProperty("BSCONFIG_IHOST");
        if (hostname == null) {
            try {
                hostname = InetAddress.getLocalHost().getHostName();
            } catch (UnknownHostException ue) {
                hostname = "unknown";
            }
        }
        return hostname;
    }

    private static PingTable pingTable(final String dc) {
        PingTable table;
        switch (dc) {
            case SAS:
                table = PingTable.SAS_TABLE;
                break;
            case MAN:
                table = PingTable.MAN_TABLE;
                break;
            case VLA:
                table = PingTable.VLA_TABLE;
                break;
            case IVA:
                table = PingTable.IVA_TABLE;
                break;
            case MYT:
                table = PingTable.MYT_TABLE;
                break;
            default:
                table = PingTable.ZERO_TABLE;
                break;
        }
        return table;
    }

    public UpstreamContext upstreamFor(final RequestInfo request) {
        return upstreams.get(request);
    }

    public AsyncClient searchClient() {
        return searchClient;
    }

    public AsyncClient indexerClient() {
        return indexerClient;
    }

    public ContentType indexerContentType() {
        return indexerContentType;
    }

    public SearchMap searchMap() {
        if (mopManager == null) {
            return searchMap;
        } else {
            return mopManager.searchMapWrapper();
        }
    }

    public MopManager mopManager() {
        return mopManager;
    }

    public static String dc(final String hostname) {
        Matcher gencfg = GENCFG_DC.matcher(hostname);
        if (gencfg.matches()) {
            return gencfg.group(1);
        }
        Matcher yp = YP_DC.matcher(hostname);
        if (yp.matches()) {
            return yp.group(1);
        }
        return "unk";
    }

    public static int pingDelay(final String hostname) {
        final String dc = dc(hostname);
        return PING_TABLE.dist(dc);
    }

    public static List<HttpHost> shuffle(
        final List<HttpHost> hosts,
        final Object o)
    {
        Collections.rotate(hosts, o.hashCode());
        return hosts;
    }

    public static List<HttpHost> localityShuffle(final List<HttpHost> hosts) {
        if (hosts.size() == 1) {
            return hosts;
        }
        final int seed = ThreadLocalRandom.current().nextInt();
        Collections.sort(
            hosts,
            new Comparator<HttpHost>() {
                //CSOFF: ReturnCount
                @Override
                public int compare(final HttpHost a, final HttpHost b) {
                    String aName = a.getHostName();
                    String bName = b.getHostName();
                    boolean aLocal =
                        aName.startsWith(SERVER_HOSTNAME)
                            || aName.startsWith(LOCALHOST);
                    boolean bLocal =
                        bName.startsWith(SERVER_HOSTNAME)
                            || bName.startsWith(LOCALHOST);
                    if (aLocal && bLocal) {
                        return aName.compareTo(bName);
                    } else if (aLocal) {
                        return -1;
                    } else if (bLocal) {
                        return 1;
                    }
                    int aDist = pingDelay(aName);
                    int bDist = pingDelay(bName);
                    int pingCmp = Integer.compare(aDist, bDist);
                    if (pingCmp != 0) {
                        return pingCmp;
                    }
                    return Integer.compare(
                        aName.hashCode() ^ seed,
                        bName.hashCode() ^ seed);
                }
                //CSON: ReturnCount
            });
        return hosts;
    }

    // Usable only for producer client and searchmap lookups
    // May contains shard number instead of real user prefix
    // In this case prefixType will be null
    public User extractUser(final ProxySession session)
        throws HttpException
    {
        String service = session.headers().get(
            YandexHeaders.SERVICE,
            null,
            NonEmptyValidator.INSTANCE);
        Prefix prefix;
        if (service == null) {
            service = session.params().get(
                SearchProxyParams.SERVICE,
                NonEmptyValidator.INSTANCE);
            Long shard = session.params().get(
                SearchProxyParams.SHARD,
                null,
                ShardNumberValidator.INSTANCE);
            if (shard == null) {
                prefix = session.params().get(
                    SearchProxyParams.PREFIX,
                    searchMap().prefixType(service));
            } else {
                prefix = new ShardPrefix(shard);
            }
        } else {
            Long shard = session.headers().get(
                YandexHeaders.ZOO_SHARD_ID,
                null,
                ShardNumberValidator.INSTANCE);
            if (shard == null) {
                prefix = session.headers().get(
                    YandexHeaders.X_SEARCH_PREFIX,
                    searchMap().prefixType(service));
            } else {
                prefix = new ShardPrefix(shard);
            }
        }
        return new User(service, prefix);
    }

    //CSOFF: MagicNumber
    private enum PingTable {
        SAS_TABLE {
            @Override
            public int dist(final String dc) {
                int dist;
                switch (dc) {
                    case MAN:
                        dist = 19500;
                        break;
                    case VLA:
                        dist = 3300;
                        break;
                    case IVA:
                        dist = 6440;
                        break;
                    case MYT:
                        dist = 7380;
                        break;
                    case SAS:
                        dist = 100;
                        break;
                    default:
                        dist = 1000;
                        break;
                }
                return dist;
            }
        },
        MAN_TABLE {
            @Override
            public int dist(final String dc) {
                int dist;
                switch (dc) {
                    case MAN:
                        dist = 100;
                        break;
                    case VLA:
                        dist = 17720;
                        break;
                    case IVA:
                        dist = 17880;
                        break;
                    case MYT:
                        dist = 17660;
                        break;
                    case SAS:
                        dist = 19780;
                        break;
                    default:
                        dist = 1000;
                        break;
                }
                return dist;
            }
        },
        VLA_TABLE {
            @Override
            public int dist(final String dc) {
                int dist;
                switch (dc) {
                    case MAN:
                        dist = 16430;
                        break;
                    case VLA:
                        dist = 100;
                        break;
                    case IVA:
                        dist = 3300;
                        break;
                    case MYT:
                        dist = 3370;
                        break;
                    case SAS:
                        dist = 3325;
                        break;
                    default:
                        dist = 1000;
                        break;
                }
                return dist;
            }
        },
        IVA_TABLE {
            @Override
            public int dist(final String dc) {
                int dist;
                switch (dc) {
                    case MAN:
                        dist = 19250;
                        break;
                    case VLA:
                        dist = 3260;
                        break;
                    case IVA:
                        dist = 100;
                        break;
                    case MYT:
                        dist = 1550;
                        break;
                    case SAS:
                        dist = 6240;
                        break;
                    default:
                        dist = 1000;
                        break;
                }
                return dist;
            }
        },
        MYT_TABLE {
            @Override
            public int dist(final String dc) {
                int dist;
                switch (dc) {
                    case MAN:
                        dist = 14800;
                        break;
                    case VLA:
                        dist = 2850;
                        break;
                    case IVA:
                        dist = 1585;
                        break;
                    case MYT:
                        dist = 100;
                        break;
                    case SAS:
                        dist = 7120;
                        break;
                    default:
                        dist = 1000;
                        break;
                }
                return dist;
            }
        },
        ZERO_TABLE {
            @Override
            public int dist(final String dc) {
                return 0;
            }
        };

        public abstract int dist(final String dc);
    }
    //CSON: MagicNumber

    @Override
    public void start() throws IOException {
        if (mopManager != null) {
            mopManager.start();
        }
        super.start();
    }

    // Fake stater which only adds lags to golovan panel
    private class LagsStatGolovanPanel implements Stater {
        @Override
        public <E extends Exception> void stats(
            final StatsConsumer<? extends E> statsConsumer)
            throws E
        {
        }

        @Override
        public void addToGolovanPanel(
            final GolovanPanel panel,
            final String statsPrefix)
        {
            String serverStatsPrefix = config.statsPrefix();
            String prefix;
            if (serverStatsPrefix == null) {
                prefix = statsPrefix;
            } else {
                prefix = serverStatsPrefix + statsPrefix;
            }

            ImmutableGolovanPanelConfig config = panel.config();
            GolovanChartGroup group = new GolovanChartGroup(prefix, prefix);
            GolovanChart lags = new GolovanChart(
                "worst-lag",
                " worst shard lag by service (ms)",
                false,
                false,
                0d);
            GolovanChart shards = new GolovanChart(
                "worst-shard",
                " worst shard number by service",
                false,
                false,
                0d);
            for (String name: searchMap().names()) {
                lags.addSignal(
                    new GolovanSignal(
                        prefix + "indexation-lag-" + name + "-max_axxx",
                        config.tag(),
                        name,
                        null,
                        0,
                        false));
                shards.addSignal(
                    new GolovanSignal(
                        prefix + "indexation-lag-" + name + "-worst-shard_axxx",
                        config.tag(),
                        name,
                        null,
                        0,
                        false));
            }
            group.addChart(lags);
            group.addChart(shards);
            panel.addCharts(
                "lags",
                null,
                group);
        }

        @Override
        public void addToAlertsConfig(
            final IniConfig alertsConfig,
            final ImmutableGolovanPanelConfig panelConfig,
            final String statsPrefix)
            throws BadRequestException
        {
            ImmutableGolovanAlertsConfig alerts = panelConfig.alerts();
            for (String name: searchMap().names()) {
                alerts.createAlert(
                    alertsConfig,
                    alerts.module()
                    + '-'
                    + GolovanAlertsConfig.clearAlertName(name)
                    + "-indexation-lag",
                    statsPrefix + "indexation-lag-" + name + "-max_axxx",
                    new AlertThresholds(1800000d, null, null));
            }
        }
    }
}

