package ru.yandex.search.proxy;

import java.io.IOException;
import java.util.List;

import org.apache.http.HttpException;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpHost;

import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.NotFoundException;
import ru.yandex.http.util.ServiceUnavailableException;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.BasicAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.request.RequestInfo;
import ru.yandex.http.util.server.HttpServer;
import ru.yandex.parser.searchmap.User;
import ru.yandex.search.request.config.SearchBackendRequestConfigBuilder;

public class UnicastHandler implements ProxyRequestHandler {
    private final SearchProxy<? extends ImmutableSearchProxyConfig> proxy;

    public UnicastHandler(
        final SearchProxy<? extends ImmutableSearchProxyConfig> proxy)
    {
        this.proxy = proxy;
    }

    @Override
    public void handle(final ProxySession session) throws HttpException {
        UpstreamContext upstream =
            proxy.upstreamFor(
                (RequestInfo) session.context().getAttribute(
                    HttpServer.REQUEST_INFO));
        String path =
            upstream.removePrefix(session.request().getRequestLine().getUri());
        User user = proxy.extractUser(session);
        BasicAsyncRequestProducerGenerator generator;
        try {
            generator = new BasicAsyncRequestProducerGenerator(
                path,
                session.request());
        } catch (IOException e) {
            throw new ServiceUnavailableException(e);
        }
        generator.copyHeader(session.request(), HttpHeaders.ACCEPT_CHARSET);
        generator.copyHeader(session.request(), HttpHeaders.ACCEPT_ENCODING);
        List<HttpHost> hosts;
        if (upstream.config().requestConfig().localityShuffle()) {
            hosts =
                SearchProxy.localityShuffle(
                    proxy.searchMap().searchHosts(user));
        } else {
            hosts =
                SearchProxy.shuffle(proxy.searchMap().searchHosts(user), user);
        }
        if (hosts.isEmpty()) {
            throw new NotFoundException("No hosts found for " + user);
        }

        SearchBackendRequestConfigBuilder requestConfig
            = new SearchBackendRequestConfigBuilder(
            session.params(),
            upstream.config().requestConfig());

        Long failoverDelay = requestConfig.failoverDelay();
//        if (failoverDelay == null) {
//            failoverDelay = DEFAULT_FAILOVER_DELAY;
//        }

        AsyncClient client = upstream.client().adjust(session.context());
        session.subscribeForCancellation(
            client.executeWithDelay(
                hosts,
                generator,
                failoverDelay,
                BasicAsyncResponseConsumerFactory.ANY_GOOD,
                session.listener().createContextGeneratorFor(client),
                new UnicastCallback(session)));
    }

    @Override
    public String toString() {
        return "Forwards request to any server containing user index. "
            + "Performs failover to mirror servers in case of failure";
    }
}

