package ru.yandex.search.request;

import java.util.Locale;

import ru.yandex.http.util.BadRequestException;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.prefix.Prefix;
import ru.yandex.util.string.StringUtils;

public class SearchRequestBuilder {

    public enum Collector {
        SORTED,
        PRUNNING,
        PASSTHROUGH;

        private final String value;

        Collector() {
            value = name().toLowerCase(Locale.ENGLISH);
        }

        public String value() {
            return value;
        }
    }

    private final QueryConstructor qc;

    public SearchRequestBuilder() {
        this(null);
    }

    public SearchRequestBuilder(final String routePostfix) {
        if (routePostfix != null) {
            qc = new QueryConstructor(StringUtils.concat("/search", routePostfix, "?"));
        } else {
            qc = new QueryConstructor("/search?");
        }
    }

    public SearchRequestBuilder dp(
        final String dpName,
        final String args,
        final String out)
        throws BadRequestException
    {
        StringBuilder sb = new StringBuilder();
        sb.append(dpName);
        sb.append('(');
        sb.append(args);
        sb.append(" ");
        sb.append(out);
        sb.append(')');
        qc.append("dp", sb.toString());
        return this;
    }

    public SearchRequestBuilder prefix(final Long prefix) {
        qc.append("prefix", prefix);
        return this;
    }

    public SearchRequestBuilder prefix(final Prefix prefix) throws BadRequestException {
        qc.append("prefix", prefix.toStringFast());
        return this;
    }

    public SearchRequestBuilder prefix(
        final String prefix)
        throws BadRequestException
    {
        qc.append("prefix", prefix);
        return this;
    }

    public SearchRequestBuilder service(
        final String service)
        throws BadRequestException
    {
        qc.append("service", service);
        return this;
    }

    public SearchRequestBuilder postfilter(
        final String left,
        final PostfilterOperator op,
        final String right)
        throws BadRequestException
    {
        qc.append("postfilter", StringUtils.concat(left, op.value(), right));
        return this;
    }

    public SearchRequestBuilder sort(
        final String field)
        throws BadRequestException
    {
        qc.append("sort", field);
        return this;
    }

    public SearchRequestBuilder group(
        final String field,
        final MergeFunc mergeFunc)
        throws BadRequestException
    {
        qc.append("group", field);
        mergeFunc(mergeFunc);
        return this;
    }

    public SearchRequestBuilder group(
        final String field)
        throws BadRequestException
    {
        qc.append("group", field);
        return this;
    }

    public SearchRequestBuilder mergeFunc(
        final MergeFunc mergeFunc)
        throws BadRequestException
    {
        qc.append("merge_func", mergeFunc.value());
        return this;
    }

    public SearchRequestBuilder collector(
        final Collector collector)
        throws BadRequestException
    {
        qc.append("collector", collector.value());
        return this;
    }

    public SearchRequestBuilder collector(
        final String collector)
        throws BadRequestException
    {
        qc.append("collector", collector);
        return this;
    }

    public SearchRequestBuilder collector(
        final Collector collector,
        final String params)
        throws BadRequestException
    {
        if (params.isEmpty()) {
            return collector(collector);
        }

        StringBuilder sb = new StringBuilder();
        sb.append(collector.value());
        sb.append('(');
        sb.append(params);
        sb.append(')');
        qc.append("collector", sb.toString());
        return this;
    }

    public SearchRequestBuilder length(final int length) {
        qc.append("length", length);
        return this;
    }

    public SearchRequestBuilder offset(final int offset) {
        qc.append("offset", offset);
        return this;
    }

    public SearchRequestBuilder get(final String fields) throws BadRequestException {
        qc.append("get", fields);
        return this;
    }

    public SearchRequestBuilder get(
        final String... fields)
        throws BadRequestException
    {
        qc.append("get", StringUtils.join(fields, ','));
        return this;
    }

    public SearchRequestQueryBuilder queryBuilder() {
        SearchRequestQueryBuilder queryBuilder = new SearchRequestQueryBuilder(this);
        return queryBuilder;
    }

    public SearchRequestBuilder query(final SearchRequestQueryBuilder query) throws BadRequestException {
        qc.append("text", query.toStringFast());
        return this;
    }

    public SearchRequestBuilder query(final String query) throws BadRequestException{
        qc.append("text", query);
        return this;
    }

    public QueryConstructor qc() {
        return qc;
    }

    public StringBuilder build() {
        return qc.sb();
    }

    @Override
    public String toString() {
        return build().toString();
    }
}
