package ru.yandex.search.request;

import ru.yandex.function.AbstractStringBuilderable;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.search.request.util.QueryAppender;
import ru.yandex.search.request.util.SearchRequestText;

public class SearchRequestQueryBuilder extends AbstractStringBuilderable {
    private final SearchRequestBuilder requestBuilder;
    private final StringBuilder textSb;

    public SearchRequestQueryBuilder(final SearchRequestBuilder requestBuilder) {
        textSb = new StringBuilder();
        this.requestBuilder = requestBuilder;
    }

    public SearchRequestQueryBuilder(final int capacity) {
        textSb = new StringBuilder(capacity);
        this.requestBuilder = null;
    }

    public SearchRequestQueryBuilder(final String str) {
        textSb = new StringBuilder(str);
        this.requestBuilder = null;
    }

    public SearchRequestQueryBuilder(final StringBuilder textSb) {
        this.textSb = textSb;
        this.requestBuilder = null;
    }

    public SearchRequestQueryBuilder fieldQuery(
        final String field,
        final String unescapedText)
    {
        return fieldQuery(field, false, unescapedText);
    }

    public SearchRequestQueryBuilder fieldQuery(
        final String field,
        final boolean wildcard,
        final String unescapedText)
    {
        textSb.append(field);
        textSb.append(':');
        textSb.append(SearchRequestText.fullEscape(unescapedText, wildcard));
        return this;
    }

    public SearchRequestQueryBuilder fieldQuery(
        final String field,
        final String... unescapedText)
    {
        return fieldQuery(field, false, QueryOperator.OR, unescapedText);
    }

    public SearchRequestQueryBuilder fieldQuery(
        final String field,
        final boolean wildcard,
        final String... unescapedText)
    {
        return fieldQuery(field, wildcard, QueryOperator.OR, unescapedText);
    }

    public SearchRequestQueryBuilder fieldQuery(
        final String field,
        final boolean wildcard,
        final QueryOperator operator,
        final String... unescapedText)
    {
        if (unescapedText.length > 1) {
            textSb.append(field);
            textSb.append(':');
            textSb.append(SearchRequestText.fullEscape(unescapedText[0], wildcard));
        } else if (unescapedText.length == 1) {
            textSb.append(field);
            textSb.append(":(");
            for (final String s: unescapedText) {
                textSb.append(SearchRequestText.fullEscape(s, wildcard));
                operator.appendTo(textSb);
            }
            textSb.setLength(textSb.length() - operator.queryValue().length());
            textSb.append(')');
        }

        return this;
    }

    public SearchRequestQueryBuilder or() {
        QueryOperator.OR.appendTo(textSb);
        return this;
    }

    public SearchRequestQueryBuilder emptyOr() {
        if (textSb.length() > 0) {
            QueryOperator.OR.appendTo(textSb);
        }

        return this;
    }

    public SearchRequestQueryBuilder and() {
        QueryOperator.OR.appendTo(textSb);
        return this;
    }

    public SearchRequestQueryBuilder append(final QueryAppender term) {
        term.appendTo(textSb);
        return this;
    }

    public SearchRequestQueryBuilder append(final String formedQuery) {
        textSb.append(formedQuery);
        return this;
    }

    public SearchRequestQueryBuilder append(final StringBuilder formedQuery) {
        textSb.append(formedQuery);
        return this;
    }

    @Override
    public String toStringFast() {
        return textSb.toString();
    }

    @Override
    public void toStringBuilder(final StringBuilder sb) {
        sb.append(textSb);
    }

    public StringBuilder textSb() {
        return textSb;
    }

    public void build() throws BadRequestException {
        if (requestBuilder != null) {
            requestBuilder.query(this);
        }
    }

}
