package ru.yandex.search.rules;

import java.nio.charset.CharacterCodingException;
import java.util.Map;
import java.util.function.Predicate;

import org.apache.http.HttpException;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.erratum.EmptyErratumResult;
import ru.yandex.erratum.ErratumClient;
import ru.yandex.erratum.ErratumResult;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.DoubleFutureCallback;
import ru.yandex.http.util.ErrorSuppressingFutureCallback;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.search.request.util.SearchRequestText;

public class MisspellRule<D, P extends RequestParams, I extends SearchInfo>
    implements SearchRule<D, P, I>
{
    private final SearchRule<D, P, I> next;
    private final ErratumClient erratumClient;
    private final String paramName;
    private final Predicate<? super D> emptyDocsPredicate;

    // TODO: Use RequestTextProvider instead of paramName
    // CSOFF: ParameterNumber
    public MisspellRule(
        final SearchRule<D, P, I> next,
        final ErratumClient erratumClient,
        final String paramName,
        final Predicate<? super D> emptyDocsPredicate)
    {
        this.next = next;
        this.erratumClient = erratumClient;
        this.paramName = paramName;
        this.emptyDocsPredicate = emptyDocsPredicate;
    }
    // CSON: ParameterNumber

    @Override
    @SuppressWarnings("FutureReturnValueIgnored")
    public void execute(final SearchRequest<D, P, I> request)
        throws HttpException
    {
        if (request.cgiParams().getBoolean("force", false)) {
            next.execute(request);
        } else {
            String text = request.cgiParams().getString(paramName, null);
            if (text != null) {
                text = SearchRequestText.normalize(text).trim();
            }
            if (text == null || text.isEmpty()) {
                next.execute(request);
            } else {
                DoubleFutureCallback<ErratumResultWithDocs, D> callback =
                    new DoubleFutureCallback<>(
                        new Callback(request, emptyDocsPredicate));
                try {
                    erratumClient.execute(
                        text,
                        request.session().listener()
                            .createContextGeneratorFor(erratumClient),
                        new ErrorSuppressingFutureCallback<>(
                            new ErratumCallback(callback.first(), request),
                            EmptyErratumResult.INSTANCE));
                } catch (CharacterCodingException e) {
                    throw new BadRequestException(
                        "Failed to encode request '" + text + '\'',
                        e);
                }
                next.execute(request.withCallback(callback.second()));
            }
        }
    }

    private class ErratumResultWithDocs {
        private final ErratumResult erratumResult;
        private final D docs;

        ErratumResultWithDocs(
            final ErratumResult erratumResult,
            final D docs)
        {
            this.erratumResult = erratumResult;
            this.docs = docs;
        }

        public ErratumResult erratumResult() {
            return erratumResult;
        }

        public D docs() {
            return docs;
        }
    }

    private class ErratumCallback
        extends AbstractFilterFutureCallback<
            ErratumResult,
            ErratumResultWithDocs>
    {
        private final SearchRequest<D, P, I> request;

        ErratumCallback(
            final FutureCallback<? super ErratumResultWithDocs> callback,
            final SearchRequest<D, P, I> request)
        {
            super(callback);
            this.request = request;
        }

        @Override
        public void completed(final ErratumResult erratumResult) {
            if (erratumResult.code() == ErratumResult.CODE_CORRECTED) {
                CgiParams cgiParams = new CgiParams(request.cgiParams());
                cgiParams.replace(paramName, erratumResult.text());
                try {
                    next.execute(
                        request
                            .withCgiParams(cgiParams)
                            .withCallback(
                                new FixedRequestCallback(
                                    callback,
                                    emptyDocsPredicate,
                                    erratumResult)));
                } catch (HttpException e) {
                    callback.failed(e);
                }
            } else {
                callback.completed(
                    new ErratumResultWithDocs(
                        EmptyErratumResult.INSTANCE,
                        null));
            }
        }
    }

    private class FixedRequestCallback
        extends AbstractFilterFutureCallback<
            D,
            ErratumResultWithDocs>
    {
        private final Predicate<? super D> emptyDocsPredicate;
        private final ErratumResult erratumResult;

        FixedRequestCallback(
            final FutureCallback<? super ErratumResultWithDocs> callback,
            final Predicate<? super D> emptyDocsPredicate,
            final ErratumResult erratumResult)
        {
            super(callback);
            this.emptyDocsPredicate = emptyDocsPredicate;
            this.erratumResult = erratumResult;
        }

        @Override
        public void completed(final D docs) {
            if (emptyDocsPredicate.test(docs)) {
                callback.completed(
                    new ErratumResultWithDocs(
                        EmptyErratumResult.INSTANCE,
                        null));
            } else {
                callback.completed(
                    new ErratumResultWithDocs(erratumResult, docs));
            }
        }
    }

    private class Callback
        extends AbstractFilterFutureCallback<
            Map.Entry<ErratumResultWithDocs, D>,
            D>
    {
        private final SearchRequest<D, P, I> request;
        private final Predicate<? super D> emptyDocsPredicate;

        Callback(
            final SearchRequest<D, P, I> request,
            final Predicate<? super D> emptyDocsPredicate)
        {
            super(request.callback());
            this.request = request;
            this.emptyDocsPredicate = emptyDocsPredicate;
        }

        @Override
        public void completed(
            final Map.Entry<ErratumResultWithDocs, D> result)
        {
            ErratumResult erratumResult = result.getKey().erratumResult();
            if (erratumResult == EmptyErratumResult.INSTANCE) {
                request.callback().completed(result.getValue());
            } else if (emptyDocsPredicate.test(result.getValue())) {
                request.searchInfo().setRequest(
                    erratumResult.text(),
                    erratumResult.rule());
                request.callback().completed(result.getKey().docs());
            } else {
                request.searchInfo().setSuggest(
                    erratumResult.text(),
                    erratumResult.rule());
                request.callback().completed(result.getValue());
            }
        }
    }
}

