package ru.yandex.search.rules;

import java.util.List;

import org.apache.http.HttpException;
import org.apache.http.protocol.HttpContext;

import ru.yandex.blackbox.BlackboxClient;
import ru.yandex.blackbox.BlackboxDbfield;
import ru.yandex.blackbox.BlackboxUserIdType;
import ru.yandex.blackbox.BlackboxUserinfo;
import ru.yandex.blackbox.BlackboxUserinfoRequest;
import ru.yandex.function.BiFunction;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.parser.uri.CgiParams;

@SuppressWarnings("FutureReturnValueIgnored")
public class ResolveUserRule<D, P extends RequestParams, I extends SearchInfo>
    implements SearchRule<D, P, I>
{
    private static final String SUID = "suid";
    private static final String UID = "uid";
    private static final String MDB = "mdb";

    private final SearchRule<D, P, I> next;
    private final BiFunction<Long, HttpContext, BlackboxClient> bboxSupplier;

    public ResolveUserRule(
        final SearchRule<D, P, I> next,
        final BiFunction<Long, HttpContext, BlackboxClient> blackboxSupplier)
    {
        this.next = next;
        this.bboxSupplier = blackboxSupplier;
    }

    @Override
    public void execute(final SearchRequest<D, P, I> request)
        throws HttpException
    {
        CgiParams params = request.cgiParams();
        Long suid = params.getLong(SUID, null);
        Long uid = params.getLong(UID, null);

        String mdb = params.getString(MDB, null);

        if (suid == null && uid == null) {
            request.callback().failed(
                new BadRequestException("No user id supplied"));
            return;
        }

        if (mdb == null) {
            BlackboxClient client;

            if (suid == null) {
                // ok uid here not null, getting suid and mdb
                client =
                    bboxSupplier.apply(uid, request.session().context());
                client.userinfo(
                    new BlackboxUserinfoRequest(uid)
                        .requiredDbfields(
                            BlackboxDbfield.MDB,
                            BlackboxDbfield.SUID),
                    request.session().listener()
                        .createContextGeneratorFor(client),
                    new BlackboxCallback(request, BlackboxUserIdType.SUID));
            } else {
                // ok suid not null, requesting mdb
                client =
                    bboxSupplier.apply(suid, request.session().context());
                client.userinfo(
                    new BlackboxUserinfoRequest(BlackboxUserIdType.SUID, suid)
                        .requiredDbfields(BlackboxDbfield.MDB),
                    request.session().listener()
                        .createContextGeneratorFor(client),
                    new BlackboxCallback(request, BlackboxUserIdType.UID));
            }
        } else {
            next.execute(request);
        }
    }

    private final class BlackboxCallback
        extends AbstractFilterFutureCallback<List<BlackboxUserinfo>, D>
    {
        private final BlackboxUserIdType type;
        private final SearchRequest<D, P, I> request;

        private BlackboxCallback(
            final SearchRequest<D, P, I> request,
            final BlackboxUserIdType type)
        {
            super(request.callback());

            this.request = request;
            this.type = type;
        }

        @Override
        public void completed(
            final List<BlackboxUserinfo> userinfos)
        {
            CgiParams params = new CgiParams(request.cgiParams());
            BlackboxUserinfo userinfo = userinfos.get(0);
            params.replace(
                MDB,
                userinfo.dbfields().get(BlackboxDbfield.MDB));

            if (type == BlackboxUserIdType.SUID) {
                params.replace(
                    SUID,
                    userinfo.dbfields().get(BlackboxDbfield.SUID));
            } else {
                params.replace(UID, String.valueOf(userinfo.uid()));
            }

            try {
                next.execute(request.withCgiParams(params));
            } catch (HttpException e) {
                failed(e);
            }
        }
    }
}
