package ru.yandex.search.rules.pure;

import java.util.function.BiFunction;

import org.apache.http.Header;
import org.apache.http.HttpException;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.blackbox.BlackboxClient;
import ru.yandex.blackbox.BlackboxDbfield;
import ru.yandex.blackbox.BlackboxOAuthException;
import ru.yandex.blackbox.BlackboxOAuthRequest;
import ru.yandex.blackbox.BlackboxUserinfo;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.parser.searchmap.User;
import ru.yandex.search.prefix.LongPrefix;
import ru.yandex.search.prefix.Prefix;
import ru.yandex.search.rules.pure.providers.BlackboxClientProvider;
import ru.yandex.search.rules.pure.providers.ProxySessionProvider;
import ru.yandex.search.rules.pure.providers.Tvm2TicketProvider;

@SuppressWarnings("FutureReturnValueIgnored")
public class ResolveOAuthUserRule<
    T extends BlackboxClientProvider
        & ProxySessionProvider
        & Tvm2TicketProvider,
    U,
    R>
    implements SearchRule<T, R>
{
    private static final String OAUTH_PREFIX = "OAuth ";

    private final ChainedSearchRule<T, U, User, R> next;
    private final String scopes;
    private final BiFunction<String, Prefix, String> serviceResolver;

    public ResolveOAuthUserRule(
        final ChainedSearchRule<T, U, User, R> next,
        final String scopes,
        final BiFunction<String, Prefix, String> serviceResolver)
    {
        this.next = next;
        this.scopes = scopes;
        this.serviceResolver = serviceResolver;
    }

    private static void requireAuthorization(
        final ProxySession session,
        final String errorMessage)
    {
        session.logger().warning(errorMessage);
        session.getResponse().addHeader(HttpHeaders.WWW_AUTHENTICATE, "OAuth");
        session.response(HttpStatus.SC_UNAUTHORIZED, errorMessage);
    }

    @Override
    public void execute(
        final T input,
        final FutureCallback<? super R> callback)
    {
        ProxySession session = input.proxySession();
        HttpRequest httpRequest = session.request();
        Header authorization =
            httpRequest.getFirstHeader(HttpHeaders.AUTHORIZATION);
        if (authorization == null) {
            requireAuthorization(session, "'Authorization' header not found");
        } else {
            String token = authorization.getValue();
            if (!token.startsWith(OAUTH_PREFIX)) {
                requireAuthorization(
                    session,
                    "Bad 'Authorization' header: '" + token
                    + "', only OAuth authorization supported");
            } else {
                token = token.substring(OAUTH_PREFIX.length());
                BlackboxOAuthRequest blackboxRequest =
                    new BlackboxOAuthRequest(token, scopes)
                        .requiredDbfields(
                            BlackboxDbfield.MDB,
                            BlackboxDbfield.SUID)
                        .addHeader(
                            YandexHeaders.X_YA_SERVICE_TICKET,
                            input.tvm2Ticket());
                Header realIp =
                    httpRequest.getFirstHeader(YandexHeaders.X_REAL_IP);
                if (realIp != null) {
                    blackboxRequest.ip(realIp.getValue());
                }
                BlackboxClient client = input.blackboxClient();
                client.oauth(
                    blackboxRequest,
                    session.listener().createContextGeneratorFor(client),
                    new Callback(callback, input));
            }
        }
    }

    private class Callback
        extends AbstractFilterFutureCallback<BlackboxUserinfo, R>
    {
        private final T input;

        Callback(
            final FutureCallback<? super R> callback,
            final T input)
        {
            super(callback);
            this.input = input;
        }

        @Override
        public void failed(final Exception e) {
            if (e instanceof BlackboxOAuthException) {
                requireAuthorization(input.proxySession(), e.getMessage());
            } else {
                super.failed(e);
            }
        }

        @Override
        public void completed(final BlackboxUserinfo userinfo) {
            try {
                long prefix;
                if (userinfo.corp()) {
                    prefix = Long.parseLong(
                        userinfo.dbfields().get(BlackboxDbfield.SUID));
                } else {
                    prefix = userinfo.uid();
                }
                Prefix prefixObject = new LongPrefix(prefix);
                next.execute(
                    input,
                    new User(
                        serviceResolver.apply(
                            userinfo.dbfields().get(BlackboxDbfield.MDB),
                            prefixObject),
                        new LongPrefix(prefix)),
                    callback);
            } catch (RuntimeException | HttpException e) {
                super.failed(e);
            }
        }
    }
}

