package ru.yandex.search.rules.pure;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.http.HttpException;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.search.rules.RequestInfo;
import ru.yandex.search.rules.RequestType;
import ru.yandex.search.rules.pure.providers.RequestProvider;

public class TranslitRule2<T extends RequestProvider, U, R>
    implements SearchRule<T, R>
{
    public static final TableSelector ACCEPT_ALL_TABLES = new TableSelector() {};
    public static final List<Table> TABLES = Collections.unmodifiableList(Arrays.asList(
        translitRuEn(),
        translitEnRu(),
        layoutRuEn(),
        layoutEnRu()));

    private final ChainedSearchRule<T, U, List<RequestInfo>, R> next;
    private final boolean includeOriginalRequest;
    private final TableSelector tableSelector;

    public TranslitRule2(
        final ChainedSearchRule<T, U, List<RequestInfo>, R> next,
        final boolean includeOriginalRequest)
    {
        this(next, includeOriginalRequest, ACCEPT_ALL_TABLES);
    }

    public TranslitRule2(
        final ChainedSearchRule<T, U, List<RequestInfo>, R> next,
        final boolean includeOriginalRequest,
        final TableSelector tableSelector)
    {
        this.next = next;
        this.includeOriginalRequest = includeOriginalRequest;
        this.tableSelector = tableSelector;
    }

    @Override
    public void execute(
        final T input,
        final FutureCallback<? super R> callback)
        throws HttpException
    {
        String request = input.request();
        Context context = new Context(request);
        Set<RequestInfo> requests = new LinkedHashSet<>(TABLES.size() << 1);
        RequestInfo original =
            new RequestInfo(RequestType.ORIGINAL, request,"");
        if (includeOriginalRequest) {
            requests.add(original);
        }
        for (Table table: TABLES) {
            if (tableSelector.useTable(request, table.name())) {
                requests.add(
                    new RequestInfo(
                        table.type(),
                        table.translate(context),
                        table.name()));
            }
        }
        if (!includeOriginalRequest) {
            requests.remove(original);
        }
        next.execute(
            input,
            Arrays.asList(requests.toArray(new RequestInfo[requests.size()])),
            callback);
    }

    private static Table translitRuEn() {
        char[] replacementTable = new char[1 << Character.SIZE];
        replacementTable['а'] = 'a';
        replacementTable['б'] = 'b';
        replacementTable['в'] = 'v';
        replacementTable['г'] = 'g';
        replacementTable['д'] = 'd';
        replacementTable['е'] = 'e';
        replacementTable['з'] = 'z';
        replacementTable['и'] = 'i';
        replacementTable['й'] = 'j';
        replacementTable['к'] = 'k';
        replacementTable['л'] = 'l';
        replacementTable['м'] = 'm';
        replacementTable['н'] = 'n';
        replacementTable['о'] = 'o';
        replacementTable['п'] = 'p';
        replacementTable['р'] = 'r';
        replacementTable['с'] = 's';
        replacementTable['т'] = 't';
        replacementTable['у'] = 'u';
        replacementTable['ф'] = 'f';
        replacementTable['х'] = 'h';
//        replacementTable['ъ'] = '`';
        replacementTable['ы'] = 'y';
//        replacementTable['ь'] = '\'';
        replacementTable['э'] = 'e';

        replacementTable['А'] = 'A';
        replacementTable['Б'] = 'B';
        replacementTable['В'] = 'V';
        replacementTable['Г'] = 'G';
        replacementTable['Д'] = 'D';
        replacementTable['Е'] = 'E';
        replacementTable['З'] = 'Z';
        replacementTable['И'] = 'I';
        replacementTable['Й'] = 'J';
        replacementTable['К'] = 'K';
        replacementTable['Л'] = 'L';
        replacementTable['М'] = 'M';
        replacementTable['Н'] = 'N';
        replacementTable['О'] = 'J';
        replacementTable['П'] = 'P';
        replacementTable['Р'] = 'R';
        replacementTable['С'] = 'S';
        replacementTable['Т'] = 'T';
        replacementTable['У'] = 'U';
        replacementTable['Ф'] = 'A';
        replacementTable['Х'] = 'H';
//        replacementTable['Ъ'] = '`';
        replacementTable['Ы'] = 'Y';
//        replacementTable['Ь'] = '\'';
        replacementTable['Э'] = 'E';
        replacementTable['«'] = '\"';
        replacementTable['»'] = '\"';

        char[][] multiReplacementTable = new char[1 << Character.SIZE][];
        multiReplacementTable['ё'] = new char[]{'y', 'o'};
        multiReplacementTable['ж'] = new char[]{'z', 'h'};
        multiReplacementTable['ц'] = new char[]{'t', 's'};
        multiReplacementTable['ч'] = new char[]{'c', 'h'};
        multiReplacementTable['ш'] = new char[]{'s', 'h'};
        multiReplacementTable['щ'] = new char[]{'s', 'h'};
        multiReplacementTable['ъ'] = new char[]{'\\', '`'};
        multiReplacementTable['ь'] = new char[]{'\\', '\''};
        multiReplacementTable['ю'] = new char[]{'y', 'u'};
        multiReplacementTable['я'] = new char[]{'y', 'a'};

        multiReplacementTable['Ё'] = new char[]{'Y', 'O'};
        multiReplacementTable['Ж'] = new char[]{'Z', 'H'};
        multiReplacementTable['Ц'] = new char[]{'T', 'S'};
        multiReplacementTable['Ч'] = new char[]{'C', 'H'};
        multiReplacementTable['Ш'] = new char[]{'S', 'H'};
        multiReplacementTable['Щ'] = new char[]{'S', 'H'};
        multiReplacementTable['Ъ'] = new char[]{'\\', '`'};
        multiReplacementTable['Ь'] = new char[]{'\\', '\''};
        multiReplacementTable['Ю'] = new char[]{'Y', 'U'};
        multiReplacementTable['Я'] = new char[]{'Y', 'A'};
        multiReplacementTable['№'] = new char[]{'N', 'o'};
        return new SingleCharTable(
            RequestType.TRANSLIT,
            "translitRuEn",
            replacementTable,
            new boolean[0],
            multiReplacementTable);
    }

    private static Table translitEnRu() {
        char[] replacementTable = new char[1 << Character.SIZE];
        replacementTable['a'] = 'а';
        replacementTable['b'] = 'б';
        replacementTable['v'] = 'в';
        replacementTable['g'] = 'г';
        replacementTable['d'] = 'д';
        replacementTable['e'] = 'е';
        replacementTable['z'] = 'з';
        replacementTable['i'] = 'и';
        replacementTable['j'] = 'й';
        replacementTable['k'] = 'к';
        replacementTable['l'] = 'л';
        replacementTable['m'] = 'м';
        replacementTable['n'] = 'н';
        replacementTable['o'] = 'о';
        replacementTable['p'] = 'п';
        replacementTable['r'] = 'р';
        replacementTable['s'] = 'с';
        replacementTable['t'] = 'т';
        replacementTable['u'] = 'у';
        replacementTable['f'] = 'ф';
        replacementTable['h'] = 'х';
        replacementTable['`'] = 'ъ';
        replacementTable['y'] = 'ы';
        replacementTable['\''] = 'ь';

        replacementTable['A'] = 'А';
        replacementTable['B'] = 'Б';
        replacementTable['V'] = 'В';
        replacementTable['G'] = 'Г';
        replacementTable['D'] = 'Д';
        replacementTable['E'] = 'Е';
        replacementTable['Z'] = 'З';
        replacementTable['I'] = 'И';
        replacementTable['J'] = 'Й';
        replacementTable['K'] = 'К';
        replacementTable['L'] = 'Л';
        replacementTable['M'] = 'М';
        replacementTable['N'] = 'Н';
        replacementTable['O'] = 'О';
        replacementTable['P'] = 'П';
        replacementTable['R'] = 'Р';
        replacementTable['S'] = 'С';
        replacementTable['T'] = 'Т';
        replacementTable['U'] = 'У';
        replacementTable['F'] = 'Ф';
        replacementTable['H'] = 'Х';
        replacementTable['Y'] = 'Ы';

        char[][] multiReplacementTable = new char[1 << Character.SIZE][];
        multiReplacementTable['x'] = new char[]{'к', 'с'};
        multiReplacementTable['X'] = new char[]{'К', 'С'};

        Map<Integer, char[]> doubleCharTable = new HashMap<>();
        doubleCharTable.put(charsToInt('y', 'o'), new char[]{'ё'});
        doubleCharTable.put(charsToInt('z', 'h'), new char[]{'ж'});
        doubleCharTable.put(charsToInt('k', 'h'), new char[]{'х'});
        doubleCharTable.put(charsToInt('t', 's'), new char[]{'ц'});
        doubleCharTable.put(charsToInt('c', 'h'), new char[]{'ч'});
        doubleCharTable.put(charsToInt('s', 'h'), new char[]{'ш'});
        doubleCharTable.put(charsToInt('y', 'u'), new char[]{'ю'});
        doubleCharTable.put(charsToInt('y', 'a'), new char[]{'я'});
        doubleCharTable.put(charsToInt('j', 'u'), new char[]{'ю'});
        doubleCharTable.put(charsToInt('j', 'a'), new char[]{'я'});

        doubleCharTable.put(charsToInt('Y', 'O'), new char[]{'Ё'});
        doubleCharTable.put(charsToInt('Z', 'H'), new char[]{'Ж'});
        doubleCharTable.put(charsToInt('K', 'H'), new char[]{'Х'});
        doubleCharTable.put(charsToInt('T', 'S'), new char[]{'Ц'});
        doubleCharTable.put(charsToInt('C', 'H'), new char[]{'Ч'});
        doubleCharTable.put(charsToInt('S', 'H'), new char[]{'Ш'});
        doubleCharTable.put(charsToInt('Y', 'U'), new char[]{'Ю'});
        doubleCharTable.put(charsToInt('Y', 'A'), new char[]{'Я'});
        doubleCharTable.put(charsToInt('J', 'U'), new char[]{'Ю'});
        doubleCharTable.put(charsToInt('J', 'A'), new char[]{'Я'});

        return new DoubleCharTable(
            RequestType.TRANSLIT,
            "translitEnRu",
            replacementTable,
            multiReplacementTable,
            doubleCharTable);
    }

    private static Table layoutRuEn() {
        char[] table = new char[1 << Character.SIZE];
        table['ё'] = '`';
        table['й'] = 'q';
        table['ц'] = 'w';
        table['у'] = 'e';
        table['к'] = 'r';
        table['е'] = 't';
        table['н'] = 'y';
        table['г'] = 'u';
        table['ш'] = 'i';
        table['щ'] = 'o';
        table['з'] = 'p';
        table['х'] = '[';
        table['ъ'] = ']';
        table['ф'] = 'a';
        table['ы'] = 's';
        table['в'] = 'd';
        table['а'] = 'f';
        table['п'] = 'g';
        table['р'] = 'h';
        table['о'] = 'j';
        table['л'] = 'k';
        table['д'] = 'l';
        table['ж'] = ';';
        table['э'] = '\'';
        table['я'] = 'z';
        table['ч'] = 'x';
        table['с'] = 'c';
        table['м'] = 'v';
        table['и'] = 'b';
        table['т'] = 'n';
        table['ь'] = 'm';
        table['б'] = ',';
        table['ю'] = '.';

        table['Ё'] = '~';
        table['Й'] = 'Q';
        table['Ц'] = 'W';
        table['У'] = 'E';
        table['К'] = 'R';
        table['Е'] = 'T';
        table['Н'] = 'Y';
        table['Г'] = 'U';
        table['Ш'] = 'I';
        table['Щ'] = 'O';
        table['З'] = 'P';
        table['Х'] = '{';
        table['Ъ'] = '}';
        table['Ф'] = 'A';
        table['Ы'] = 'S';
        table['В'] = 'D';
        table['А'] = 'F';
        table['П'] = 'G';
        table['Р'] = 'H';
        table['О'] = 'J';
        table['Л'] = 'K';
        table['Д'] = 'L';
        table['Ж'] = ':';
        table['Э'] = '"';
        table['Я'] = 'Z';
        table['Ч'] = 'X';
        table['С'] = 'C';
        table['М'] = 'V';
        table['И'] = 'B';
        table['Т'] = 'N';
        table['Ь'] = 'M';
        table['Б'] = '<';
        table['Ю'] = '>';

        //Do not convert "Парахин" to "Gfhf[by"
        //because '[' will break request to two tokens
        //wich will cause too many false positive results
        boolean[] exceptionsTable = new boolean[1 << Character.SIZE];
        exceptionsTable['ё'] = true;
        exceptionsTable['х'] = true;
        exceptionsTable['ъ'] = true;
        exceptionsTable['ж'] = true;
        exceptionsTable['э'] = true;
        exceptionsTable['б'] = true;
        exceptionsTable['ю'] = true;

        exceptionsTable['Ё'] = true;
        exceptionsTable['Х'] = true;
        exceptionsTable['Ъ'] = true;
        exceptionsTable['Ж'] = true;
        exceptionsTable['Э'] = true;
        exceptionsTable['Б'] = true;
        exceptionsTable['Ю'] = true;

        return new SingleCharTable(
            RequestType.KEYBOARD_LAYOUT,
            "layoutRuEn",
            table,
            exceptionsTable,
            new char[0][]);
    }

    private static Table layoutEnRu() {
        char[] table = new char[1 << Character.SIZE];
        table['`'] = 'ё';
        table['q'] = 'й';
        table['w'] = 'ц';
        table['e'] = 'у';
        table['r'] = 'к';
        table['t'] = 'е';
        table['y'] = 'н';
        table['u'] = 'г';
        table['i'] = 'ш';
        table['o'] = 'щ';
        table['p'] = 'з';
        table['['] = 'х';
        table[']'] = 'ъ';
        table['a'] = 'ф';
        table['s'] = 'ы';
        table['d'] = 'в';
        table['f'] = 'а';
        table['g'] = 'п';
        table['h'] = 'р';
        table['j'] = 'о';
        table['k'] = 'л';
        table['l'] = 'д';
        table[';'] = 'ж';
        table['\''] = 'э';
        table['z'] = 'я';
        table['x'] = 'ч';
        table['c'] = 'с';
        table['v'] = 'м';
        table['b'] = 'и';
        table['n'] = 'т';
        table['m'] = 'ь';
        table[','] = 'б';
        table['.'] = 'ю';

        table['~'] = 'Ё';
        table['Q'] = 'Й';
        table['W'] = 'Ц';
        table['E'] = 'У';
        table['R'] = 'К';
        table['T'] = 'Е';
        table['Y'] = 'Н';
        table['U'] = 'Г';
        table['I'] = 'Ш';
        table['O'] = 'Щ';
        table['P'] = 'З';
        table['['] = 'Х';
        table[']'] = 'Ъ';
        table['A'] = 'Ф';
        table['S'] = 'Ы';
        table['D'] = 'В';
        table['F'] = 'А';
        table['G'] = 'П';
        table['H'] = 'Р';
        table['J'] = 'О';
        table['K'] = 'Л';
        table['L'] = 'Д';
        table[':'] = 'Ж';
        table['"'] = 'Э';
        table['Z'] = 'Я';
        table['X'] = 'Ч';
        table['C'] = 'С';
        table['V'] = 'М';
        table['B'] = 'И';
        table['N'] = 'Т';
        table['M'] = 'Ь';
        table['<'] = 'Б';
        table['>'] = 'Ю';
        return new SingleCharTable(
            RequestType.TRANSLIT,
            "layoutEnRu",
            table,
            new boolean[0],
            new char[0][]);
    }

    public static class Context {
        private final String request;
        private final char[] requestChars;
        private final StringBuilder sb;

        public Context(final String request) {
            this.request = request;
            requestChars = request.toCharArray();
            sb = new StringBuilder(request.length() << 1);
        }
    }

    public interface Table {
        String translate(Context context);

        String name();

        RequestType type();
    }

    public static class SingleCharTable implements Table {
        private final RequestType type;
        private final String name;
        private final char[] replacementTable;
        private final char replacementOffset;
        private final char replacementMax;
        private final boolean[] exceptionsTable;
        private final char exceptionsOffset;
        private final char exceptionsMax;
        private final char[][] multiReplacementTable;
        private final char multiReplacementOffset;
        private final char multiReplacementMax;

        SingleCharTable(
            final RequestType type,
            final String name,
            final char[] replacementTable,
            final boolean[] exceptionsTable,
            final char[][] multiReplacementTable)
        {
            this.name = name;
            this.type = type;
            int replacementOffset = 0;
            int replacementLength = 0;
            for (int i = 0; i < replacementTable.length; ++i) {
                if (replacementTable[i] != 0) {
                    if (replacementOffset == 0) {
                        replacementOffset = i;
                    }
                    replacementLength = i - replacementOffset + 1;
                }
            }
            this.replacementTable = new char[replacementLength];
            for (int i = 0; i < replacementLength; ++i) {
                this.replacementTable[i] =
                    replacementTable[i + replacementOffset];
            }
            this.replacementOffset = (char) replacementOffset;
            replacementMax = (char) (replacementOffset + replacementLength);

            int exceptionsOffset = 0;
            int exceptionsLength = 0;
            for (int i = 0; i < exceptionsTable.length; ++i) {
                if (exceptionsTable[i]) {
                    if (exceptionsOffset == 0) {
                        exceptionsOffset = i;
                    }
                    exceptionsLength = i - exceptionsOffset + 1;
                }
            }
            this.exceptionsTable = new boolean[exceptionsLength];
            for (int i = 0; i < exceptionsLength; ++i) {
                this.exceptionsTable[i] =
                    exceptionsTable[i + exceptionsOffset];
            }
            this.exceptionsOffset = (char) exceptionsOffset;
            exceptionsMax = (char) (exceptionsOffset + exceptionsLength);

            int multiReplacementOffset = 0;
            int multiReplacementLength = 0;
            for (int i = 0; i < multiReplacementTable.length; ++i) {
                if (multiReplacementTable[i] != null) {
                    if (multiReplacementOffset == 0) {
                        multiReplacementOffset = i;
                    }
                    multiReplacementLength = i - multiReplacementOffset + 1;
                }
            }
            this.multiReplacementTable = new char[multiReplacementLength][];
            for (int i = 0; i < multiReplacementLength; ++i) {
                this.multiReplacementTable[i] =
                    multiReplacementTable[i + multiReplacementOffset];
            }
            this.multiReplacementOffset = (char) multiReplacementOffset;
            multiReplacementMax =
                (char) (multiReplacementOffset + multiReplacementLength);
        }

        protected boolean translateChar(final char c, final StringBuilder sb) {
            if (c >= exceptionsOffset
                    && c < exceptionsMax
                    && exceptionsTable[c - exceptionsOffset])
            {
                sb.append(c);
                return false;
            }
            char replacement;
            if (c < replacementOffset || c >= replacementMax) {
                replacement = 0;
            } else {
                replacement = replacementTable[c - replacementOffset];
            }
            boolean translated;
            if (replacement == 0) {
                char[] replacements;
                if (c < multiReplacementOffset || c >= multiReplacementMax) {
                    replacements = null;
                } else {
                    replacements =
                        multiReplacementTable[c - multiReplacementOffset];
                }
                if (replacements == null) {
                    sb.append(c);
                    translated = false;
                } else {
                    for (char replaced: replacements) {
                        sb.append(replaced);
                    }
                    translated = true;
                }
            } else {
                sb.append(replacement);
                translated = true;
            }
            return translated;
        }

        @Override
        public String translate(final Context context) {
            StringBuilder sb = context.sb;
            sb.setLength(0);
            boolean translated = false;
            for (char c: context.requestChars) {
                if (translateChar(c, sb)) {
                    translated = true;
                }
            }
            if (translated) {
                return sb.toString();
            } else {
                return context.request;
            }
        }

        @Override
        public String name() {
            return name;
        }

        @Override
        public RequestType type() {
            return type;
        }
    }

    private static int charsToInt(final char first, final char second) {
        return first | (second << Character.SIZE);
    }

    private static class DoubleCharTable extends SingleCharTable {
        private final Map<Integer, char[]> doubleCharTable;

        DoubleCharTable(
            final RequestType type,
            final String name,
            final char[] replacementTable,
            final char[][] multiReplacementTable,
            final Map<Integer, char[]> doubleCharTable)
        {
            super(
                type,
                name,
                replacementTable,
                new boolean[0],
                multiReplacementTable);
            this.doubleCharTable = doubleCharTable;
        }

        @Override
        public String translate(final Context context) {
            StringBuilder sb = context.sb;
            sb.setLength(0);
            char[] request = context.requestChars;
            boolean translated = false;
            for (int i = 0; i < request.length; ++i) {
                char c = request[i];
                int next = i + 1;
                if (next < request.length) {
                    char[] replacement =
                        doubleCharTable.get(charsToInt(c, request[next]));
                    if (replacement == null) {
                        if (translateChar(c, sb)) {
                            translated = true;
                        }
                    } else {
                        for (char replaced: replacement) {
                            sb.append(replaced);
                        }
                        translated = true;
                        i++;
                    }
                } else {
                    if (translateChar(c, sb)) {
                        translated = true;
                    }
                }
            }
            if (translated) {
                return sb.toString();
            } else {
                return context.request;
            }
        }
    }

    public interface TableSelector {
        default boolean useTable(final String request, final String table) {
            return true;
        }
    }
}

