package ru.yandex.parser.searchmap;

import java.io.BufferedReader;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.StringReader;
import java.nio.charset.StandardCharsets;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Function;

import org.apache.http.HttpHost;

import ru.yandex.search.prefix.PrefixType;

public class SearchMap implements Function<User, SearchMapShard> {
    public static final long SHARDS_COUNT = 65534L;
    public static final SearchMapShard EMPTY_SHARD =
        new SearchMapShard(Collections.emptyList(), -1);

    private final ImmutableSearchMapConfig config;
    protected volatile Map<String, SearchMapRow> map;
    private final AtomicInteger version = new AtomicInteger(0);

    public SearchMap(final ImmutableSearchMapConfig config)
        throws IOException, ParseException
    {
        this.config = config;
        if (config.file() == null) {
            map = load(new BufferedReader(new StringReader(config.content())));
        } else {
            reload();
        }
    }

    public PrefixType prefixType() {
        return config.prefixType();
    }

    public PrefixType prefixType(final String service) {
        SearchMapRow row = map.get(service);
        if (row == null) {
            return prefixType();
        } else {
            return row.prefixType();
        }
    }

    public boolean reloadable() {
        return config.file() != null;
    }

    public int version() {
        return version.get();
    }

    public void reload() throws IOException, ParseException {
        try (BufferedReader reader = new BufferedReader(new InputStreamReader(
                new FileInputStream(config.file()), StandardCharsets.UTF_8)))
        {
            map = load(reader);
            version.incrementAndGet();
        }
    }

    private Map<String, SearchMapRow> load(
        final BufferedReader reader)
        throws IOException, ParseException
    {
        SearchMapParser parser = new SearchMapParser(config);
        String line;
        int lineNumber = 0;
        while ((line = reader.readLine()) != null) {
            ++lineNumber;
            if (line.isEmpty() || line.charAt(0) == '#') {
                continue;
            }

            parser.parseLine(lineNumber, line);
        }
        return parser.build();
    }

    @Override
    public SearchMapShard apply(final User user) {
        SearchMapRow row = map.get(user.service());
        if (row == null
            || (row.prefixType() != user.prefix().type()
                && user.prefix().type() != null))
        {
            return EMPTY_SHARD;
        } else {
            return row.get((int) (user.prefix().hash() % SHARDS_COUNT));
        }
    }

    public SearchMapRow row(final String service) {
        return map.get(service);
    }

    public SearchMapShard hosts(final User user) {
        return apply(user);
    }

    public Set<SearchMapHost> hosts() {
        Set<SearchMapHost> hosts = new HashSet<>();
        for (SearchMapRow row: map.values()) {
            for (int i = 0; i < SHARDS_COUNT; ++i) {
                hosts.addAll(row.get(i));
            }
        }
        return hosts;
    }

    public List<HttpHost> searchHosts(final User user) {
        List<SearchMapHost> hosts = hosts(user);
        int size = hosts.size();
        List<HttpHost> searchHosts = new ArrayList<>(size);
        for (int i = 0; i < size; ++i) {
            HttpHost searchHost = hosts.get(i).searchHost();
            if (searchHost != null) {
                searchHosts.add(searchHost);
            }
        }
        return searchHosts;
    }

    public Set<HttpHost> searchHosts() {
        Set<HttpHost> hosts = new HashSet<>();
        for (SearchMapRow row: map.values()) {
            for (int i = 0; i < SHARDS_COUNT; ++i) {
                for (SearchMapHost host: row.get(i)) {
                    if (host.searchHost() != null) {
                        hosts.add(host.searchHost());
                    }
                }
            }
        }
        return hosts;
    }

    public List<HttpHost> indexerHosts(final User user) {
        List<SearchMapHost> hosts = hosts(user);
        int size = hosts.size();
        List<HttpHost> indexerHosts = new ArrayList<>(size);
        for (int i = 0; i < size; ++i) {
            HttpHost indexerHost = hosts.get(i).indexerHost();
            if (indexerHost != null) {
                indexerHosts.add(indexerHost);
            }
        }
        return indexerHosts;
    }

    public Set<HttpHost> indexerHosts() {
        Set<HttpHost> hosts = new HashSet<>();
        for (SearchMapRow row: map.values()) {
            for (int i = 0; i < SHARDS_COUNT; ++i) {
                for (SearchMapHost host: row.get(i)) {
                    if (host.indexerHost() != null) {
                        hosts.add(host.indexerHost());
                    }
                }
            }
        }
        return hosts;
    }

    public SearchMapShard shardGet(
        final Integer shardnum,
        final String service
    )
    {
        SearchMapRow row = map.get(service);
        if (row == null) {
            return EMPTY_SHARD;
        }
        return row.get(shardnum);
    }

    public Set<List<ZooKeeperAddress>> zkHosts(final String service) {
        Set<List<ZooKeeperAddress>> zkHosts = new HashSet<>();
        SearchMapRow row = map.get(service);
        if (row == null) {
            return Collections.emptySet();
        }
        for (int i = 0; i < SHARDS_COUNT; ++i) {
            SearchMapShard shard = row.get(i);
            if (shard != null) {
                List<ZooKeeperAddress> zk = shard.zk();
                if (!zk.isEmpty()) {
                    zkHosts.add(zk);
                }
            }
        }
        Set<Set<ZooKeeperAddress>> uniqHosts = new HashSet<>();
        for (List<ZooKeeperAddress> hosts: zkHosts) {
            uniqHosts.add(new HashSet<>(hosts));
        }
        zkHosts.clear();
        for (Set<ZooKeeperAddress> hosts: uniqHosts) {
            zkHosts.add(new ArrayList<>(hosts));
        }
        return zkHosts;
    }

    public Set<String> names() {
        return Collections.unmodifiableSet(map.keySet());
    }

    public PrefixInfo prefixInfo(final User user) {
        return new PrefixInfo(user, hosts(user));
    }
}

