package ru.yandex.parser.searchmap;

import java.util.List;
import java.util.Locale;

import ru.yandex.parser.string.BooleanParser;
import ru.yandex.parser.string.EnumParser;
import ru.yandex.search.prefix.PrefixType;

public class SearchMapLine {
    private static final EnumParser<PrefixType> PREFIX_TYPE_PARSER =
        new EnumParser<>(PrefixType.class);

    private final List<String> services;
    private final int iNum;
    private final String host;
    private final int minShard;
    private final int maxShard;
    private final int searchPort;
    private final int indexerPort;
    private final PrefixType prefixType;
    private final String dc;
    private final String zk;

    public SearchMapLine(
        final String line,
        final ImmutableSearchMapConfig config,
        final SearchMapParser interner)
        throws Exception
    {
        int space = line.indexOf(' ');
        if (space == -1) {
            throw new IllegalArgumentException("No space delimiter found");
        }
        services = interner.parseServices(line, 0, space);
        String iNum = null;
        String host = null;
        String shards = null;
        String skipSearch = null;
        String searchPort = null;
        String indexerPort = null;
        String prefixType = null;
        String dc = null;
        String zk = null;
        int pos = space + 1;
        while (pos != 0) {
            int comma = line.indexOf(',', pos);
            String param;
            if (comma == -1) {
                param = line.substring(pos);
            } else {
                param = line.substring(pos, comma);
            }
            pos = comma + 1;
            int colon = param.indexOf(':');
            if (colon != -1) {
                String name = param.substring(0, colon);
                String value = param.substring(colon + 1);
                switch (name) {
                    case "iNum":
                        iNum = value;
                        break;
                    case "host":
                        host = value;
                        break;
                    case "shards":
                        shards = value;
                        break;
                    case "skip_search":
                        skipSearch = value;
                        break;
                    case "prefix_type":
                        prefixType = value;
                        break;
                    case "dc":
                        dc = value;
                        break;
                    case "zk":
                        zk = value;
                        break;
                    default:
                        if (name.equals(config.searchPortName())) {
                            searchPort = value;
                        } else if (
                            name.equals(config.fallbackSearchPortName()))
                        {
                            if (searchPort == null) {
                                searchPort = value;
                            }
                        } else if (name.equals(config.indexerPortName())) {
                            indexerPort = value;
                        } else if (
                            name.equals(config.fallbackIndexerPortName()))
                        {
                            if (indexerPort == null) {
                                indexerPort = value;
                            }
                        }
                        break;
                }
            }
        }
        if (iNum == null) {
            this.iNum = -1;
        } else {
            this.iNum = Integer.parseInt(iNum);
        }
        if (host == null) {
            throw new IllegalArgumentException("Host is not set");
        }
        if (shards == null) {
            throw new IllegalArgumentException("Shards is not set");
        }
        this.host = interner.intern(host.toLowerCase(Locale.ENGLISH));
        int dash = shards.indexOf('-');
        minShard = Math.max(Integer.parseInt(shards.substring(0, dash)), 0);
        maxShard = Math.min(
            Integer.parseInt(shards.substring(dash + 1)),
            (int) (SearchMap.SHARDS_COUNT - 1));
        boolean skipSearchValue = false;
        if (skipSearch != null && !config.ignoreSkipSearch()) {
            skipSearchValue = BooleanParser.INSTANCE.apply(skipSearch);
        }
        if (searchPort == null || skipSearchValue) {
            this.searchPort = -1;
        } else {
            this.searchPort = Integer.parseInt(searchPort);
        }
        if (indexerPort == null) {
            this.indexerPort = -1;
        } else {
            this.indexerPort = Integer.parseInt(indexerPort);
        }
        if (prefixType == null) {
            this.prefixType = config.prefixType();
        } else {
            this.prefixType = PREFIX_TYPE_PARSER.apply(prefixType);
        }
        this.dc = dc;
        this.zk = zk;
    }

    public List<String> services() {
        return services;
    }

    public int iNum() {
        return iNum;
    }

    public String host() {
        return host;
    }

    public int minShard() {
        return minShard;
    }

    public int maxShard() {
        return maxShard;
    }

    public int searchPort() {
        return searchPort;
    }

    public int indexerPort() {
        return indexerPort;
    }

    public PrefixType prefixType() {
        return prefixType;
    }

    public String dc() {
        return dc;
    }

    public String zk() {
        return zk;
    }
}

