package ru.yandex.parser.searchmap;

import java.text.ParseException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.Map;

import ru.yandex.function.StringProcessor;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.NonEmptyValidator;

public class SearchMapParser {
    private static final CollectionParser<
        ZooKeeperAddress,
        List<ZooKeeperAddress>,
        RuntimeException>
        ZK_PARSER = new CollectionParser<>(
            ZooKeeperAddressParser.INSTANCE,
            ArrayList::new,
            '|');
    private static final CollectionParser<String, List<String>, Exception>
        SERVICES_PARSER =
            new CollectionParser<>(NonEmptyValidator.INSTANCE, ArrayList::new);
    private static final int PORTS_COUNT = 65536;

    private final Map<String, String> stringsCache = new HashMap<>();
    private final Integer[] intsCache = new Integer[PORTS_COUNT];
    private final Map<String, List<ZooKeeperAddress>> zkCache =
        new HashMap<>();
    @SuppressWarnings("IdentityHashMapBoxing")
    private final IdentityHashMap<
        Integer,
        IdentityHashMap<Integer, IdentityHashMap<String, SearchMapHost>>>
        hostsCache = new IdentityHashMap<>();
    private final StringProcessor<List<ZooKeeperAddress>, RuntimeException>
        zkParser = new StringProcessor<>(ZK_PARSER);
    private final StringProcessor<List<String>, Exception> servicesParser =
        new StringProcessor<>(SERVICES_PARSER);
    private final Map<String, SearchMapRow> searchMap = new HashMap<>();
    private final ImmutableSearchMapConfig config;

    public SearchMapParser(final ImmutableSearchMapConfig config) {
        this.config = config;
    }

    public String intern(final String string) {
        String cached = stringsCache.get(string);
        if (cached == null) {
            cached = string.intern();
            stringsCache.put(cached, cached);
        }
        return cached;
    }

    private Integer intern(final int integer) {
        Integer result;
        if (integer >= 0 && integer < PORTS_COUNT) {
            result = intsCache[integer];
            if (result == null) {
                result = integer;
                intsCache[integer] = result;
            }
        } else {
            result = null;
        }
        return result;
    }

    private List<ZooKeeperAddress> internZk(final String zk) {
        List<ZooKeeperAddress> cached;
        if (zk == null) {
            cached = Collections.emptyList();
        } else {
            cached = zkCache.get(zk);
            if (cached == null) {
                cached = zkParser.process(zk);
                zkCache.put(zk, cached);
            }
        }
        return cached;
    }

    public List<String> parseServices(
        final String line,
        final int off,
        final int len)
        throws Exception
    {
        return servicesParser.process(line, off, len);
    }

    public void parseLine(final int lineNumber, final String line)
        throws ParseException
    {
        try {
            parseLine(line);
        } catch (Throwable t) {
            ParseException e = new ParseException(
                "Failed to parse line: " + line,
                lineNumber);
            e.initCause(t);
            throw e;
        }
    }

    @SuppressWarnings("IdentityHashMapBoxing")
    private void parseLine(final String rawLine) throws Exception {
        SearchMapLine line = new SearchMapLine(rawLine, config, this);
        SearchMapHost host = hostsCache
            .computeIfAbsent(
                intern(line.searchPort()),
                x -> new IdentityHashMap<>())
            .computeIfAbsent(
                intern(line.indexerPort()),
                x -> new IdentityHashMap<>())
            .computeIfAbsent(line.host(), x -> new SearchMapHost(line));
        List<ZooKeeperAddress> zk = internZk(line.zk());
        List<String> services = line.services();
        int size = services.size();
        for (int i = 0; i < size; ++i) {
            String service = intern(services.get(i));
            SearchMapRow row = searchMap.get(service);
            if (row == null) {
                row = new SearchMapRow(line.prefixType());
                searchMap.put(service, row);
            } else if (row.prefixType() != line.prefixType()) {
                throw new IllegalArgumentException(
                    "Prefix type mismatch. Expected: " + row.prefixType()
                    + ", got: " + line.prefixType());
            }
            for (int j = line.minShard(); j <= line.maxShard(); ++j) {
                row.add(j, zk, host, line.iNum());
            }
        }
    }

    public Map<String, SearchMapRow> build() {
        Map<SearchMapShard, SearchMapShard> shardsCache = new HashMap<>();
        Map<SearchMapRow, SearchMapRow> rowsCache = new HashMap<>();
        for (Map.Entry<String, SearchMapRow> entry: searchMap.entrySet()) {
            SearchMapRow row = entry.getValue();
            for (int i = (int) SearchMap.SHARDS_COUNT; i-- > 0;) {
                SearchMapShard shard = row.get(i);
                if (shard == null) {
                    row.replace(i, SearchMap.EMPTY_SHARD);
                } else {
                    SearchMapShard cached = shardsCache.get(shard);
                    if (cached == null) {
                        shardsCache.put(shard, shard);
                    } else {
                        row.replace(i, cached);
                    }
                }
            }
            SearchMapRow cached = rowsCache.get(row);
            if (cached == null) {
                rowsCache.put(row, row);
            } else {
                entry.setValue(cached);
            }
        }
        return searchMap;
    }
}

