package ru.yandex.search.mail.shivaka;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import ru.yandex.http.config.HttpTargetConfig;
import ru.yandex.http.config.HttpTargetConfigBuilder;
import ru.yandex.http.util.server.AbstractHttpProxyConfigBuilder;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.parser.searchmap.SearchMapConfig;
import ru.yandex.parser.searchmap.SearchMapConfigBuilder;
import ru.yandex.parser.string.NonEmptyValidator;

public abstract class AbstractShivakaConfigBuilder
    <T extends AbstractShivakaConfigBuilder<T>>
    extends AbstractHttpProxyConfigBuilder<T>
    implements ShivakaConfig
{
    private SearchMapConfigBuilder searchMapConfig;
    private HttpTargetConfigBuilder queueConfig;

    private long statUpdateDelay;
    private String queueSuffix;

    private String filterHostPattern;

    private Map<String, String> localMonitoringConfig;
    private Map<String, String> localConsumerServiceConfig;

    protected AbstractShivakaConfigBuilder(final ShivakaConfig config) {
        super(config);
        searchMapConfig(config.searchMapConfig());
        queueConfig(config.queueConfig());
        statUpdateDelay(config.statUpdateDelay());
        queueSuffix(config.queueSuffix());
        localMonitoringConfig(config.localMonitoringConfig());
        localConsumerServiceConfig(config.localConsumerServiceConfig());
    }

    protected AbstractShivakaConfigBuilder(
        final IniConfig config,
        final ShivakaConfig defaults)
        throws ConfigException
    {
        super(config, defaults);
        searchMapConfig = new SearchMapConfigBuilder(
                config.section("searchmap"),
                defaults.searchMapConfig());
        queueConfig = new HttpTargetConfigBuilder(
                config.section("queue"),
                defaults.queueConfig());

        IniConfig shivakaConfig = config.section("shivaka");

        statUpdateDelay = STAT_UPDATE_DELAY.extract(
                shivakaConfig,
                defaults.statUpdateDelay());

        queueSuffix = QUEUE_SUFFIX.extract(
                shivakaConfig,
                defaults.queueSuffix());

        filterHostPattern = FILTER_HOST_PATTERN.extract(
                shivakaConfig,
                defaults.filterHostPattern());

        IniConfig monitoringConfig
            = config.sectionOrNull("monitoring_template");

        if (monitoringConfig == null) {
            localMonitoringConfig(defaults.localMonitoringConfig());
        } else {
            Set<String> keys = monitoringConfig.keys();
            localMonitoringConfig = new HashMap<>(keys.size() << 1);
            for (String key: keys) {
                String from;
                try {
                    from = NonEmptyValidator.INSTANCE.apply(key);
                } catch (Exception e) {
                    throw new ConfigException(
                        "Failed to parse template mapping " + key,
                        e);
                }
                localMonitoringConfig.put(
                    from,
                    monitoringConfig.get(key, NonEmptyValidator.INSTANCE));
            }
        }

        IniConfig consumerServiceConfig
            = config.sectionOrNull("consumer_service");

        if (consumerServiceConfig == null) {
            localConsumerServiceConfig(defaults.localConsumerServiceConfig());
        } else {
            Set<String> keys = consumerServiceConfig.keys();
            localConsumerServiceConfig = new HashMap<>(keys.size() << 1);
            for (String key: keys) {
                String from;
                try {
                    from = NonEmptyValidator.INSTANCE.apply(key);
                } catch (Exception e) {
                    throw new ConfigException(
                        "Failed to parse service to consumer mapping " + key,
                        e);
                }
                localConsumerServiceConfig.put(
                    from,
                    consumerServiceConfig.get(key, NonEmptyValidator.INSTANCE));
            }
        }
    }

    @Override
    public SearchMapConfigBuilder searchMapConfig() {
        return searchMapConfig;
    }

    public T searchMapConfig(final SearchMapConfig searchMapConfig) {
        this.searchMapConfig = new SearchMapConfigBuilder(searchMapConfig);
        return self();
    }

    @Override
    public HttpTargetConfigBuilder queueConfig() {
        return queueConfig;
    }

    public T queueConfig(final HttpTargetConfig queueConfig) {
        this.queueConfig = new HttpTargetConfigBuilder(queueConfig);
        return self();
    }

    @Override
    public long statUpdateDelay() {
        return statUpdateDelay;
    }

    public T statUpdateDelay(final long startupDelay) {
        this.statUpdateDelay = startupDelay;
        return self();
    }

    @Override
    public String queueSuffix() {
        return queueSuffix;
    }

    public T queueSuffix(final String queueSuffix) {
        this.queueSuffix = queueSuffix;
        return self();
    }

    @Override
    public Map<String, String> localMonitoringConfig() {
        return localMonitoringConfig;
    }

    public T localMonitoringConfig(
        final Map<String,
            String> localMonitoringConfig
    )
    {
        this.localMonitoringConfig = new HashMap<>(localMonitoringConfig);
        return self();
    }

    @Override
    public Map<String, String> localConsumerServiceConfig() {
        return localConsumerServiceConfig;
    }

    public T localConsumerServiceConfig(
        final Map<String,
            String> localConsumerServiceConfig
    )
    {
        this.localConsumerServiceConfig
            = new HashMap<>(localConsumerServiceConfig);
        return self();
    }

    @Override
    public String filterHostPattern() {
        return filterHostPattern;
    }

    public T filterHostPattern(final String filterHostPattern) {
        this.filterHostPattern = filterHostPattern;
        return self();
    }
}

