package ru.yandex.search.mail.shivaka;

import java.io.IOException;
import java.text.ParseException;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Timer;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import org.apache.http.HttpException;
import org.apache.http.HttpStatus;

import ru.yandex.collection.Pattern;
import ru.yandex.http.proxy.HttpProxy;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.json.writer.JsonType;
import ru.yandex.parser.searchmap.SearchMap;
import ru.yandex.parser.searchmap.SearchMapHost;
import ru.yandex.parser.searchmap.ZooKeeperAddress;
import ru.yandex.search.mail.shivaka.handlers.ShivakaQueueInfoHandler;
import ru.yandex.search.mail.shivaka.handlers.ShivakaTemplateHandler;
import ru.yandex.search.mail.shivaka.tasks.RenewStatTask;
import ru.yandex.search.mail.shivaka.tools.QueuelenStatProvider;
import ru.yandex.stater.Stater;
import ru.yandex.stater.StatsConsumer;

public class Shivaka
    extends HttpProxy<ImmutableShivakaConfig>
    implements ProxyRequestHandler
{
    private static final int MAX_POOL_SIZE = 10;
    private static final String TEST = "/test/";
    private static final Integer STARTDELAY = 100;

    private final Timer timer;
    private final AsyncClient queueClient;
    private final QueuelenStatProvider queuelenStat;
    private final SearchMap searchMap;
    private final HashSet<HashSet<ZooKeeperAddress>> zkHostsUniq;
    private final HashSet<String> smHostsSet;
    private final java.util.regex.Pattern filterHostPattern;

    private final ThreadPoolExecutor taskpool = new ThreadPoolExecutor(
        1,
        1,
        1,
        TimeUnit.HOURS,
        new ArrayBlockingQueue<Runnable>(MAX_POOL_SIZE));

    private Map<String, // service
            Map<Integer, // shard
                    Map<Integer, // inum
                            Map<String, // hostname
                                    QueuelenString>>>> filtersearchmap; // data

    private final Map<String,
        Map<HashSet<ZooKeeperAddress>,
            SearchmapQueue>> mapinfo;

    public Shivaka(final ImmutableShivakaConfig config)
        throws HttpException, IOException, ParseException
    {
        super(config);

        this.logger().info("Shivaka starting.");
        this.logger().info(
            "Shivaka queuelen suffix is: " + config.queueSuffix());

        this.logger().info(
            "Сonsumer-service config: "
            + JsonType.NORMAL.toString(
                config.localConsumerServiceConfig()));

        this.logger().info(
            "Monitoring templates: "
            + JsonType.NORMAL.toString(
                config.localMonitoringConfig()));

        this.filterHostPattern = java.util.regex.Pattern.compile(config.filterHostPattern());

        this.logger().info(
                "Shivaka filterHostPattern is: " + config.filterHostPattern());

        this.queueClient = client("Queue", config.queueConfig());
        this.timer = new Timer(getName() + "-Timer", true);
        timer.schedule(
            new RenewStatTask(this),
            STARTDELAY,
            config.statUpdateDelay());

        this.searchMap = config.searchMapConfig().build();

        this.queuelenStat = new QueuelenStatProvider(this);
        this.zkHostsUniq = new HashSet<>();

        this.smHostsSet = new HashSet<>(searchMap().hosts().size());
        // TODO: make different hostset for all services separately
        for (SearchMapHost host : searchMap().hosts()) {
            // Expect that searchmap ALWAYS contains json_indexer_port
            try {
                smHostsSet.add(host.indexerHost().getHostName());
            } catch (NullPointerException e) {
                this.logger()
                    .info("WARNING: json_indexer_port invalid,"
                        + "indexer host is null: "
                        + host.toString());
                e.printStackTrace();
            }
        }

        this.filtersearchmap = new ConcurrentHashMap<>();

        Map<String,
            Map<HashSet<ZooKeeperAddress>,
                SearchmapQueue>> mapinfo = new LinkedHashMap<>();

        for (String service : searchMap().names()) {
            this.logger().info("Services found: " + service);
            // Create hashmap for service:<zookeeper hostset>:<bitset>
            Map<HashSet<ZooKeeperAddress>,
                    SearchmapQueue> serviceinfo =
                    mapinfo.computeIfAbsent(
                        service,
                        k -> new LinkedHashMap<>()
                    );

            for (int shardnum = 0;
                 shardnum < SearchMap.SHARDS_COUNT;
                 ++shardnum)
            {
                HashSet<ZooKeeperAddress> currentZk = new HashSet<>(searchMap()
                    .shardGet(shardnum, service)
                    .zk());

                if (!currentZk.isEmpty()) {
                    // Add current zks to zkHostlist, uniq hosts
                    // this.zkHostsUniq.add(currentZk);
                    this.zkHostsUniq.add(currentZk);
                } else {
                    continue;
                }

                SearchmapQueue currqueue = serviceinfo.get(currentZk);

                if (currqueue != null) {
                    currqueue.turnOnShard(shardnum);
                } else {
                    serviceinfo.put(
                            currentZk,
                            new SearchmapQueue()
                                    .turnOnShard(shardnum));
                }
            }
        }

        this.mapinfo = Collections.unmodifiableMap(mapinfo);

        this.logger().info("Queue's found in searchmap: "
                + zkHostsUniq);

        register(new Pattern<>(TEST, true), this);

        register(new Pattern<>("/renewstat", false),
                new ShivakaQueueInfoHandler(this));

        registerStater(new ShivaksAsyncServerStater(this));

        register(new Pattern<>("/template", true),
            new ShivakaTemplateHandler(this));
    }

    public AsyncClient queueClient() {
        return queueClient;
    }

    public SearchMap searchMap() {
        return searchMap;
    }

    public void executeTask(final Runnable task) {
        taskpool.execute(task);
    }

    public QueuelenStatProvider queuelenStat() {
        return queuelenStat;
    }

    public java.util.regex.Pattern getFilterHostPattern(){
        return filterHostPattern;
    }

    public HashSet<HashSet<ZooKeeperAddress>> zkHostsUniq() {
        return zkHostsUniq;
    }

    public HashSet<String> smHostsSet() {
        return smHostsSet;
    }

    public Map<String,
            Map<Integer,
                    Map<Integer,
                            Map<String, QueuelenString>>>> filtersearchmap()
    {
        return filtersearchmap;
    }

    public Map<String,
            Map<HashSet<ZooKeeperAddress>,
                    SearchmapQueue>> mapinfo()
    {
        return mapinfo;
    }

    @Override
    public void handle(final ProxySession session) throws HttpException {
        // Do we need create context here?
        ShivakaRequestContext context = new ShivakaRequestContext(
                this,
                session);

        defaulthandle(context);
    }

    private void defaulthandle(final ShivakaRequestContext context)
        throws HttpException
    {
        context.session().response(
            HttpStatus.SC_OK,
            this.searchMap.hosts().toString());
    }

    private static class ShivaksAsyncServerStater implements Stater {
        private final Shivaka shivaka;

        ShivaksAsyncServerStater(final Shivaka shivaka) {
            this.shivaka = shivaka;
        }

        @Override
        public <E extends Exception> void stats(
                final StatsConsumer<? extends E> statsConsumer)
                throws E
        {
            for (String service: this.shivaka.queuelenStat()
                    .getQueuestat()
                    .keySet())
            {
                shivaka.logger().info("from stat hashmap service: "
                        + service);
                for (Map.Entry<String, long[]> sstat
                        : this.shivaka.queuelenStat()
                        .getQueuestat()
                        .get(service)
                        .entrySet())
                {
                    shivaka.logger().info("key: "
                            + sstat.getKey()
                            + " value: "
                            + sstat.getValue()[0]);
                    statsConsumer.stat(
                            sstat.getKey(),
                            sstat.getValue()[0]
                    );
                }
            }
        }
    }
}
