package ru.yandex.search.mail.shivaka.handlers;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.collection.LazyList;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.ErrorSuppressingFutureCallback;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.nio.AsyncStringConsumerFactory;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.parser.searchmap.ZooKeeperAddress;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.mail.shivaka.QueuelenString;
import ru.yandex.search.mail.shivaka.SearchmapQueue;
import ru.yandex.search.mail.shivaka.Shivaka;

public class ShivakaQueueInfoHandler implements ProxyRequestHandler {
    private static final long[] BAD_VALUE = {1L};
    private static final long[] GOOD_VALUE = {0L};
    private static final String DELIMITER = "-";
    private static final String BAD_SIGNAL_SUFFIX = "_bad_signal_axxx";
    //private static final String BAD_QUEUE_SIGNAL_SUFFIX = "_bad_queue_axxx";
    private static final String NEGATIVE_QUEUE_SIGNAL_SUFFIX
        = "_negative_queue_axxx";
    private final Shivaka shivaka;
    private ConcurrentHashMap<String, long[]> badSignalCheck;
    private ConcurrentHashMap<String, long[]> badQueueCheck;
    private ConcurrentHashMap<String, long[]> negativeQueueCheck;

    public ShivakaQueueInfoHandler(final Shivaka shivaka) {
        this.shivaka = shivaka;
        this.badSignalCheck = new ConcurrentHashMap<>();
        this.badQueueCheck = new ConcurrentHashMap<>();
        this.negativeQueueCheck = new ConcurrentHashMap<>();
    }

    private ConcurrentHashMap<String, long[]> getBadSignalCheck() {
        return this.badSignalCheck;
    }

    private ConcurrentHashMap<String, long[]> getBadQueueCheck() {
        return this.badQueueCheck;
    }

    private ConcurrentHashMap<String, long[]> getNegativeQueueCheck() {
        return this.negativeQueueCheck;
    }

    @Override
    public void handle(
            final ProxySession session)
            throws HttpException, IOException
    {
        this.badSignalCheck.clear();
        this.badQueueCheck.clear();
        this.negativeQueueCheck.clear();

        // We doesnot set bad value for consumer services
        // because we doesnt know - in wich queue consumer service exist
        // Example: if base name disk_queue_disk then
        // disk_queue_ reindex_consumer - name for consumer
        // TODO: specify in config queue if it have consumer service?
        // TODO: if service have consumer and bad status - push good val?
        for (String service : shivaka.mapinfo().keySet()) {
            shivaka.logger().info("Make empty stats for queue: " + service);
            shivaka.filtersearchmap()
                .computeIfAbsent(
                    service,
                    k -> new LinkedHashMap<>());

            this.getBadSignalCheck().put(
                service,
                new long[]{BAD_VALUE[0]});
        }

        MultiFutureCallback<QueuelenResult> mfcb =
                new MultiFutureCallback<>(new StatsCalcCallback(session));

        AsyncClient client =
                shivaka.queueClient().adjust(session.context());

        QueryConstructor qc =
                new QueryConstructor(this.shivaka.config().queueSuffix());

        for (HashSet<ZooKeeperAddress> zkShard
                : shivaka.zkHostsUniq())
        {
            List<HttpHost> zkHostlist = new LazyList<>(
                    new ArrayList<>(zkShard),
                zooKeeperAddress -> zooKeeperAddress.host()
            );
            client.execute(
                    zkHostlist,
                    new BasicAsyncRequestProducerGenerator(qc.toString()),
                    AsyncStringConsumerFactory.OK,
                    session.listener().adjustContextGenerator(
                            client.httpClientContextGenerator()),
                    new QueuelenParser(
                        new ErrorSuppressingFutureCallback<>(
                            mfcb.newCallback(),
                            new QueuelenResult(zkShard, true)),
                            new HashSet<>(zkShard))
            );
        }
        mfcb.done();
    }

    final class QueuelenResult {
        private HashSet<ZooKeeperAddress> zk;
        private boolean error;

        private QueuelenResult(
            final HashSet<ZooKeeperAddress> zk,
            final boolean error)
        {
            this.zk = zk;
            this.error = error;
            //shivaka.logger().info("Queue parse completed: " + this.zk);
        }

        public HashSet<ZooKeeperAddress> getZk() {
            shivaka.logger().info("Queuelen result called");
            return zk;
        }
    }

    private final class QueuelenParser
            extends AbstractFilterFutureCallback<String, QueuelenResult>
    {
        private final HashSet<ZooKeeperAddress> zk;

        private QueuelenParser(
                final FutureCallback<QueuelenResult> callback,
                final HashSet<ZooKeeperAddress> zk)
        {
            super(callback);
            this.zk = zk;
        }

        @Override
        public void completed(final String result) {
            shivaka.logger().info("Queuelen querying for zklist:" + zk);

            try (BufferedReader rdr =
                    new BufferedReader(new StringReader(result)))
            {
                for (String line = rdr.readLine();
                     line != null;
                     line = rdr.readLine())
                {
                    //shivaka.logger().info("LINE IS" + line);
                    QueuelenString qs = new QueuelenString(line);
                    String serviceName = qs.service();
                    String serviceNameOrig = qs.service();

                    if (shivaka.getFilterHostPattern().matcher(qs.hostname()).matches()) {
                         // shivaka.logger().info("Filter host from queuelen due filterHostPattern policy: "
                         //       + qs.hostName() );
                        continue;
                    }


                    // determine servicename
                    // (different for consumer configured service)

                    if (shivaka.config()
                        .localConsumerServiceConfig()
                        .get(qs.hostname()) != null)
                    {
                        serviceName = qs.service().concat(DELIMITER)
                            + shivaka.config()
                            .localConsumerServiceConfig()
                            .get(qs.hostname());
                    }

                    Map<HashSet<ZooKeeperAddress>,
                            SearchmapQueue> serviceStatus = shivaka
                            .mapinfo()
                            .get(qs.service());

                    if (serviceStatus == null) {
                        continue;
                    }

                    // Simply increase negative value for queue if
                    // shard queue is negative.
                    // We doesnt check shardStatus here and all
                    // negative shards counting as error

                    ShivakaQueueInfoHandler
                        .this
                        .getNegativeQueueCheck()
                        .computeIfAbsent(
                            serviceName,
                            k -> new long[]{GOOD_VALUE[0]});

                    if (qs.shard() < 0) {
                        shivaka.logger().info("Negative shard found: "
                            + qs.shard().toString()
                            + " queueSize: "
                            + qs.consumerQueueSize());

                        ShivakaQueueInfoHandler
                            .this
                            .getNegativeQueueCheck()
                            .get(serviceName)[0] += qs.consumerQueueSize();
                        continue;
                    }

                    Boolean shardStatus = serviceStatus.get(zk)
                            .getShardInfo(qs.shard());

                    if (shardStatus) {
                        // If shard status is true - we need write information
                        // from shard, for hosts existing in searchmap.
                        if (shivaka.smHostsSet().contains(qs.hostname())
                            || (shivaka.config()
                            .localConsumerServiceConfig()
                            .get(qs.hostname()) != null))
                        {
                            // If service exists into sm - put good val
                            // by default
                            ShivakaQueueInfoHandler
                                .this
                                .getBadQueueCheck().computeIfAbsent(
                                    serviceName,
                                    k -> new long[]{GOOD_VALUE[0]});

                            // push also good value to service original name
                            // if there consumer
                            ShivakaQueueInfoHandler
                                .this
                                .getBadQueueCheck().computeIfAbsent(
                                    serviceNameOrig,
                                    k -> new long[]{GOOD_VALUE[0]});

                            if (qs.consumerQueueSize() < 0) {
                                shivaka.logger().warning("Some shard in"
                                    + " queue is negative! "
                                    + " service: "
                                    + serviceName
                                    + " shard: "
                                    + qs.shard().toString()
                                    + " queuesize: "
                                    + qs.consumerQueueSize()
                                );
                                ShivakaQueueInfoHandler
                                    .this
                                    .getBadQueueCheck().put(
                                    serviceName,
                                    new long[]{BAD_VALUE[0]});
                            }

                            Integer inum = shivaka.searchMap()
                                .shardGet(qs.shard(),
                                    qs.service())
                                .iNum();


                            synchronized (shivaka.filtersearchmap())
                            {
                                shivaka.filtersearchmap()
                                        .computeIfAbsent(
                                                serviceName,
                                                k -> new LinkedHashMap<>())
                                        .computeIfAbsent(
                                                qs.shard(),
                                                k -> new LinkedHashMap<>())
                                        .computeIfAbsent(
                                                inum,
                                                k -> new LinkedHashMap<>())
                                        .put(qs.hostname(), qs);
                            }
                        }
                    }
                }
            } catch (IOException e) {
                callback.failed(e);
                return;
            }
            callback.completed(new QueuelenResult(zk, false));
        }
    }

    private final class StatsCalcCallback
            extends AbstractProxySessionCallback<List<QueuelenResult>>
    {
        private StatsCalcCallback(final ProxySession session) {
            super(session);
        }

        @Override
        public void completed(final List<QueuelenResult> result) {
            for (QueuelenResult qr : result) {
                if (qr.error) {
                    shivaka.logger().warning("Queue stats cannot be parsed: "
                        + qr.getZk());
                    for (
                        Map.Entry<String,
                            Map<HashSet<ZooKeeperAddress>,
                                SearchmapQueue>> srvInfo
                            : shivaka.mapinfo().entrySet())
                    {
                        if (srvInfo.getValue().keySet().contains(qr.getZk())) {
                            shivaka.logger().warning("Cannot get full "
                                + "info for service (set bad): "
                                + srvInfo.getKey()
                            );
                            ShivakaQueueInfoHandler
                                .this
                                .getBadQueueCheck().put(
                                    srvInfo.getKey(),
                                new long[]{BAD_VALUE[0]});
                        }
                    }
                } else {
                    shivaka.logger().info("Queue stats was parsed: "
                        + qr.getZk());
                }
            }

            // Full stat for bad signals with queuelen info,
            // by default - BAD_VALUE for all services.
            for (Map.Entry<String, long[]> srvBadQ
                : ShivakaQueueInfoHandler.this.getBadQueueCheck().entrySet())
            {
                ShivakaQueueInfoHandler
                    .this
                    .getBadSignalCheck()
                    .put(
                        srvBadQ.getKey(),
                        srvBadQ.getValue());
            }

            // So we have "service": value sigs, we need push it to service
            // with another value

            for (Map.Entry<String, long[]> chkSignal
                : ShivakaQueueInfoHandler.this.getBadSignalCheck().entrySet())
            {
                shivaka.queuelenStat()
                    .getQueuestat()
                    .computeIfAbsent(chkSignal.getKey(),
                        k -> new LinkedHashMap<>())
                    .put(chkSignal.getKey() + BAD_SIGNAL_SUFFIX,
                        chkSignal.getValue());
            }

            for (Map.Entry<String, long[]> chkSignal
                : ShivakaQueueInfoHandler
                .this
                .getNegativeQueueCheck()
                .entrySet())
            {
                shivaka.queuelenStat()
                    .getQueuestat()
                    .computeIfAbsent(chkSignal.getKey(),
                        k -> new LinkedHashMap<>())
                    .put(chkSignal.getKey() + NEGATIVE_QUEUE_SIGNAL_SUFFIX,
                        chkSignal.getValue());
            }

            shivaka.logger().info("Renew queue stats.");
            shivaka.executeTask(
                new RenewTask(new RenewTaskCallback(session),
                shivaka));
        }
    }

    private final class RenewTaskCallback
        extends AbstractProxySessionCallback<Void>
    {
        private RenewTaskCallback(final ProxySession session) {
            super(session);
        }

        @Override
        public void completed(final Void result) {
            shivaka.logger().info("Queue stats renewed.");
            session.response(
                HttpStatus.SC_OK,
                "Queue stats renewed.\r\n"
            );
        }
    }

    private static final class RenewTask implements Runnable {
        private final RenewTaskCallback callback;
        private final Shivaka shivaka;

        private RenewTask(
            final RenewTaskCallback callback,
            final Shivaka shivaka)
        {
            this.callback = callback;
            this.shivaka = shivaka;
        }

        @Override
        public void run() {
            try {
                shivaka.queuelenStat().renewstat();
                callback.completed(null);
            } catch (Exception e) {
                callback.failed(e);
            }
        }
    }
}
