package ru.yandex.search.mail.shivaka.tools;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import ru.yandex.search.mail.shivaka.QueuelenString;
import ru.yandex.search.mail.shivaka.Shivaka;

public class QueuelenStatProvider {
    private static final Integer SHARDS_TOP_MAX = 5;
    private static final Integer DEFAULT_QUEUE_VALUE = -1;
    private static final Long DEFAULT_MON_VALUE = 0L;

    private static final String AXXX = "_axxx";

    private Map<String, Map<String, long[]>> queuestat;
    private Map<String, Map<Integer, long[]>> shardstatQueue;
    private Map<String, ArrayList<ArrayList<Long>>> shardstatQueueList;
    private Map<String, Map<Integer, long[]>> shardstatConsumerTimeLag;
    private Map<String,
        ArrayList<ArrayList<Long>>> shardstatConsumerTimeLagList;
    private final Shivaka shivaka;

    public QueuelenStatProvider(final Shivaka shivaka) {
        this.shivaka = shivaka;

        // Signals key is name, value - signal value
        this.queuestat = new ConcurrentHashMap<>();
        // Shard statistics service, shard, min position for shard
        this.shardstatQueue = new LinkedHashMap<>();
        this.shardstatQueueList = new HashMap<>();
        this.shardstatConsumerTimeLag = new LinkedHashMap<>();
        this.shardstatConsumerTimeLagList = new LinkedHashMap<>();
    }

    public synchronized void renewstat() {
        this.shardstatQueue.clear();
        this.shardstatConsumerTimeLag.clear();

        //  shardstatQueue - shard queue statistics
        // ( min value for all hosts in shard)
        //  shardstatConsumerTimestamp - consumer timestamp statistics
        // ( min value for all hosts in shard)
        for (
            Map.Entry<String,
                    Map<Integer,
                            Map<Integer,
                                    Map<String,
                                            QueuelenString>>>> serviceinfo
                : shivaka.filtersearchmap().entrySet())
        {
            shivaka.logger().info("Operating service: "
                + serviceinfo.getKey()
            );

            // Full consumer and queue min maps statistics if it doesnot exists
            Map<Integer, long[]> currQueueStat =
                    this.shardstatQueue.computeIfAbsent(
                        serviceinfo.getKey(),
                        k -> new LinkedHashMap<>()
                    );

            Map<Integer, long[]> currConsumerTimeLagStat =
                this.shardstatConsumerTimeLag.computeIfAbsent(
                    serviceinfo.getKey(),
                    k -> new LinkedHashMap<>()
                );

            currQueueStat.putIfAbsent(
                0,
                new long[]{DEFAULT_QUEUE_VALUE});
            currConsumerTimeLagStat.putIfAbsent(
                0,
                new long[]{DEFAULT_QUEUE_VALUE});

            for (
                Map.Entry<Integer,
                        Map<Integer,
                                Map<String,
                                        QueuelenString>>> shardResult
                    : serviceinfo.getValue().entrySet())
            {
                long[] currQueueShard =
                    currQueueStat.computeIfAbsent(
                        shardResult.getKey(),
                        k -> new long[]{DEFAULT_QUEUE_VALUE});

                long[] currConsumerTimeLagShard =
                    currConsumerTimeLagStat.computeIfAbsent(
                        shardResult.getKey(),
                        k -> new long[]{DEFAULT_QUEUE_VALUE}
                    );

                for (
                    Map.Entry<Integer,
                            Map<String,
                                    QueuelenString>> inumResult
                        : shardResult.getValue().entrySet())
                {
                    for (
                        Map.Entry<String,
                                QueuelenString> qresult
                            : inumResult.getValue().entrySet())
                    {
                        // Iterate through all hosts in shard, get stats
                        // and write it to specified hashmap
                        QueuelenString cs = qresult.getValue();

                        if ((currQueueShard[0] == DEFAULT_QUEUE_VALUE)
                            || (cs.consumerQueueSize()
                            < currQueueShard[0]))
                        {
                            currQueueShard[0] = cs.consumerQueueSize();
                        }

                        if ((currConsumerTimeLagShard[0]
                            == DEFAULT_QUEUE_VALUE)
                            || (cs.consumerTimeLag()
                            < currConsumerTimeLagShard[0]))
                        {
                            currConsumerTimeLagShard[0]
                                = cs.consumerTimeLag();
                        }
                    }
                }
            }
        }

        this.shardstatQueueList = this.sortedList(shardstatQueue);
        this.shardstatConsumerTimeLagList = this.sortedList(
            shardstatConsumerTimeLag
        );
        //this.shardstatQueueList.clear();
        this.makeStats();
    }

    public Map<String, Map<String, long[]>> getQueuestat() {
        return queuestat;
    }

    private void makeStats() {
        /*for (
            Map.Entry<String, ArrayList<ArrayList<Long>>> serviceinfo
                : this.shardstatConsumerTimeLagList.entrySet()
        )
        {
            for (ArrayList<Long> currlist : serviceinfo.getValue()) {
                shivaka.logger().info("Service: "
                    + serviceinfo.getKey()
                    + " Consumer lag list: "
                    + currlist);
            }
        }*/

        // Make simple queuelen by services
        for (Map.Entry<String, ArrayList<ArrayList<Long>>> serviceinfo
            : shardstatQueueList.entrySet())
        {
            Map<String, long[]> currstat =
                this.queuestat.computeIfAbsent(
                    serviceinfo.getKey(),
                    k -> new LinkedHashMap<>()
                );

            long[] minkey = new long[1];

            for (
                ArrayList<Long> shardinfo : serviceinfo.getValue())
            {
                // Filter negative default queue values ( by default = -1
                // for new shards)
                //if (DEFAULT_QUEUE_VALUE != shardinfo.get(1).intValue()) {
                minkey[0] += shardinfo.get(1);
                //}
            }
            //shivaka.logger().info("SERVICE NAME IS"
            //    + serviceinfo.getKey());
            currstat.put(serviceinfo.getKey() + "-total-queue_axxx",
                minkey);
        }

        for (Map.Entry<String, ArrayList<ArrayList<Long>>> serviceinfo
            : shardstatQueueList.entrySet())
        {
            Map<String, long[]> currstat =
                this.queuestat.computeIfAbsent(
                    serviceinfo.getKey(),
                    k -> new LinkedHashMap<>()
                );

            int maxtopshards;
            if (serviceinfo.getValue().size() < SHARDS_TOP_MAX) {
                maxtopshards = serviceinfo.getValue().size();
            } else {
                maxtopshards = SHARDS_TOP_MAX;
            }

            for (int i = 0; i < maxtopshards; ++i) {
                long[] statkey = new long[1];
                statkey[0] = serviceinfo.getValue().get(i).get(1);
                currstat.put(serviceinfo.getKey()
                        + "-biggest-shard-queue-"
                        + i
                        + AXXX,
                    statkey);
                statkey = new long[1];
                statkey[0] = serviceinfo.getValue().get(i).get(0);
                currstat.put(serviceinfo.getKey()
                        + "-shardnum-biggest-shard-queue-"
                        + i
                        + AXXX,
                    statkey);
            }
        }

        for (Map.Entry<String, ArrayList<ArrayList<Long>>> serviceinfo
            : shardstatConsumerTimeLagList.entrySet())
        {
            Map<String, long[]> currstat =
                this.queuestat.computeIfAbsent(
                    serviceinfo.getKey(),
                    k -> new LinkedHashMap<>()
                );

            int maxtopshards;
            if (serviceinfo.getValue().size() < SHARDS_TOP_MAX) {
                maxtopshards = serviceinfo.getValue().size();
            } else {
                maxtopshards = SHARDS_TOP_MAX;
            }

            for (int i = 0; i < maxtopshards; ++i) {
                long[] statkey = new long[1];
                statkey[0] = serviceinfo.getValue().get(i).get(1);
                currstat.put(serviceinfo.getKey()
                        + "-biggest-consumer-pos-"
                        + i
                        + AXXX,
                    statkey);
                statkey = new long[1];
                statkey[0] = serviceinfo.getValue().get(i).get(0);
                currstat.put(serviceinfo.getKey()
                        + "-shardnum-biggest-consumer-pos-"
                        + i
                        + AXXX,
                    statkey);
            }
        }

        for (Map.Entry<String, Map<String, long[]>> serviceinfo
            : queuestat.entrySet())
        {
            shivaka.logger().info("from stat hashmap service: "
                + serviceinfo.getKey());
            for (Map.Entry<String, long[]> sstat
                : serviceinfo.getValue().entrySet())
            {
                shivaka.logger().info("key: "
                    + sstat.getKey()
                    + " value: "
                    + sstat.getValue()[0]);
            }
        }
    }

    private Map<String,
        ArrayList<ArrayList<Long>>> sortedList(
            final Map<String, Map<Integer, long[]>> shardstat
    )
    {
        // Make arraylist of list for shards for all services
        Map<String,
            ArrayList<ArrayList<Long>>> shardstatList = new HashMap<>();

        for (Map.Entry<String, Map<Integer, long[]>> serviceinfo
            : shardstat.entrySet())
        {
            ArrayList<ArrayList<Long>> currstatList
                = shardstatList.computeIfAbsent(
                    serviceinfo.getKey(),
                k -> new ArrayList<>()
            );

            for (
                Map.Entry<Integer,
                    long[]> shardinfo : serviceinfo.getValue().entrySet())
            {
                ArrayList<Long> shardinfoList = new ArrayList<>();
                shardinfoList.add(Long.valueOf(shardinfo.getKey()));
                if (shardinfo.getValue()[0] < 0) {
                    // If we cut consumer host from queue,
                    // we have queue without hosts with default -1 value
                    // for all shards, pass it
                    shivaka.logger().warning("Negative value detected: "
                        + serviceinfo.getKey()
                        + " shard: "
                        + shardinfo.getKey()
                        + " shard value: "
                        + shardinfo.getValue()[0]);
                    shardinfoList.add(DEFAULT_MON_VALUE);
                } else {
                    shardinfoList.add(shardinfo.getValue()[0]);
                }

                currstatList.add(shardinfoList);
            }

            currstatList.sort((arg0, arg1)
                -> arg1.get(1).compareTo(arg0.get(1)));
        }

        return shardstatList;
    }
}
