package ru.yandex.search.translit;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

public class Translit {
    public static final Translit INSTANCE = new Translit();
    public static final TranslitTableSelector ACCEPT_ALL_TABLES = new TranslitTableSelector() {};
    public static final List<TranslitTable> TABLES = Collections.unmodifiableList(Arrays.asList(
        translitRuEn(),
        translitEnRu(),
        layoutRuEn(),
        layoutEnRu()));

    private final TranslitTableSelector tableSelector;

    public Translit() {
        this(ACCEPT_ALL_TABLES);
    }

    public Translit(final TranslitTableSelector tableSelector) {
        this.tableSelector = tableSelector;
    }

    public Set<String> process(final String request) {
        return process(request, true, tableSelector);
    }

    public static Set<String> process(
        final String request,
        final boolean includeOriginal)
    {
        return process(request, includeOriginal, ACCEPT_ALL_TABLES);
    }

    public static Set<String> process(
        final String request,
        final boolean includeOriginal,
        final TranslitTableSelector selector)
    {
        Set<String> requests = new LinkedHashSet<>(TABLES.size() << 1);
        if (includeOriginal) {
            requests.add(request);
        }
        TranslitContext context = new TranslitContext(request);

        for (TranslitTable table: TABLES) {
            if (selector.useTable(request, table.name())) {
                requests.add(table.translate(context));
            }
        }
        if (!includeOriginal) {
            requests.remove(request);
        }

        return requests;
    }

    private static TranslitTable translitRuEn() {
        char[] replacementTable = new char[1 << Character.SIZE];
        replacementTable['а'] = 'a';
        replacementTable['б'] = 'b';
        replacementTable['в'] = 'v';
        replacementTable['г'] = 'g';
        replacementTable['д'] = 'd';
        replacementTable['е'] = 'e';
        replacementTable['з'] = 'z';
        replacementTable['и'] = 'i';
        replacementTable['й'] = 'j';
        replacementTable['к'] = 'k';
        replacementTable['л'] = 'l';
        replacementTable['м'] = 'm';
        replacementTable['н'] = 'n';
        replacementTable['о'] = 'o';
        replacementTable['п'] = 'p';
        replacementTable['р'] = 'r';
        replacementTable['с'] = 's';
        replacementTable['т'] = 't';
        replacementTable['у'] = 'u';
        replacementTable['ф'] = 'f';
        replacementTable['х'] = 'h';
//        replacementTable['ъ'] = '`';
        replacementTable['ы'] = 'y';
//        replacementTable['ь'] = '\'';
        replacementTable['э'] = 'e';

        replacementTable['А'] = 'A';
        replacementTable['Б'] = 'B';
        replacementTable['В'] = 'V';
        replacementTable['Г'] = 'G';
        replacementTable['Д'] = 'D';
        replacementTable['Е'] = 'E';
        replacementTable['З'] = 'Z';
        replacementTable['И'] = 'I';
        replacementTable['Й'] = 'J';
        replacementTable['К'] = 'K';
        replacementTable['Л'] = 'L';
        replacementTable['М'] = 'M';
        replacementTable['Н'] = 'N';
        replacementTable['О'] = 'J';
        replacementTable['П'] = 'P';
        replacementTable['Р'] = 'R';
        replacementTable['С'] = 'S';
        replacementTable['Т'] = 'T';
        replacementTable['У'] = 'U';
        replacementTable['Ф'] = 'A';
        replacementTable['Х'] = 'H';
//        replacementTable['Ъ'] = '`';
        replacementTable['Ы'] = 'Y';
//        replacementTable['Ь'] = '\'';
        replacementTable['Э'] = 'E';
        replacementTable['«'] = '\"';
        replacementTable['»'] = '\"';

        char[][] multiReplacementTable = new char[1 << Character.SIZE][];
        multiReplacementTable['ё'] = new char[]{'y', 'o'};
        multiReplacementTable['ж'] = new char[]{'z', 'h'};
        multiReplacementTable['ц'] = new char[]{'t', 's'};
        multiReplacementTable['ч'] = new char[]{'c', 'h'};
        multiReplacementTable['ш'] = new char[]{'s', 'h'};
        multiReplacementTable['щ'] = new char[]{'s', 'h'};
        multiReplacementTable['ъ'] = new char[]{'\\', '`'};
        multiReplacementTable['ь'] = new char[]{'\\', '\''};
        multiReplacementTable['ю'] = new char[]{'y', 'u'};
        multiReplacementTable['я'] = new char[]{'y', 'a'};

        multiReplacementTable['Ё'] = new char[]{'Y', 'O'};
        multiReplacementTable['Ж'] = new char[]{'Z', 'H'};
        multiReplacementTable['Ц'] = new char[]{'T', 'S'};
        multiReplacementTable['Ч'] = new char[]{'C', 'H'};
        multiReplacementTable['Ш'] = new char[]{'S', 'H'};
        multiReplacementTable['Щ'] = new char[]{'S', 'H'};
        multiReplacementTable['Ъ'] = new char[]{'\\', '`'};
        multiReplacementTable['Ь'] = new char[]{'\\', '\''};
        multiReplacementTable['Ю'] = new char[]{'Y', 'U'};
        multiReplacementTable['Я'] = new char[]{'Y', 'A'};
        multiReplacementTable['№'] = new char[]{'N', 'o'};
        return new TranslitSingleCharTable(
            "translitRuEn",
            replacementTable,
            new boolean[0],
            multiReplacementTable);
    }

    private static TranslitTable translitEnRu() {
        char[] replacementTable = new char[1 << Character.SIZE];
        replacementTable['a'] = 'а';
        replacementTable['b'] = 'б';
        replacementTable['v'] = 'в';
        replacementTable['g'] = 'г';
        replacementTable['d'] = 'д';
        replacementTable['e'] = 'е';
        replacementTable['z'] = 'з';
        replacementTable['i'] = 'и';
        replacementTable['j'] = 'й';
        replacementTable['k'] = 'к';
        replacementTable['l'] = 'л';
        replacementTable['m'] = 'м';
        replacementTable['n'] = 'н';
        replacementTable['o'] = 'о';
        replacementTable['p'] = 'п';
        replacementTable['r'] = 'р';
        replacementTable['s'] = 'с';
        replacementTable['t'] = 'т';
        replacementTable['u'] = 'у';
        replacementTable['f'] = 'ф';
        replacementTable['h'] = 'х';
        replacementTable['`'] = 'ъ';
        replacementTable['y'] = 'ы';
        replacementTable['\''] = 'ь';

        replacementTable['A'] = 'А';
        replacementTable['B'] = 'Б';
        replacementTable['V'] = 'В';
        replacementTable['G'] = 'Г';
        replacementTable['D'] = 'Д';
        replacementTable['E'] = 'Е';
        replacementTable['Z'] = 'З';
        replacementTable['I'] = 'И';
        replacementTable['J'] = 'Й';
        replacementTable['K'] = 'К';
        replacementTable['L'] = 'Л';
        replacementTable['M'] = 'М';
        replacementTable['N'] = 'Н';
        replacementTable['O'] = 'О';
        replacementTable['P'] = 'П';
        replacementTable['R'] = 'Р';
        replacementTable['S'] = 'С';
        replacementTable['T'] = 'Т';
        replacementTable['U'] = 'У';
        replacementTable['F'] = 'Ф';
        replacementTable['H'] = 'Х';
        replacementTable['Y'] = 'Ы';

        char[][] multiReplacementTable = new char[1 << Character.SIZE][];
        multiReplacementTable['x'] = new char[]{'к', 'с'};
        multiReplacementTable['X'] = new char[]{'К', 'С'};

        Map<Integer, char[]> doubleCharTable = new HashMap<>();
        doubleCharTable.put(charsToInt('y', 'o'), new char[]{'ё'});
        doubleCharTable.put(charsToInt('z', 'h'), new char[]{'ж'});
        doubleCharTable.put(charsToInt('k', 'h'), new char[]{'х'});
        doubleCharTable.put(charsToInt('t', 's'), new char[]{'ц'});
        doubleCharTable.put(charsToInt('c', 'h'), new char[]{'ч'});
        doubleCharTable.put(charsToInt('s', 'h'), new char[]{'ш'});
        doubleCharTable.put(charsToInt('y', 'u'), new char[]{'ю'});
        doubleCharTable.put(charsToInt('y', 'a'), new char[]{'я'});
        doubleCharTable.put(charsToInt('j', 'u'), new char[]{'ю'});
        doubleCharTable.put(charsToInt('j', 'a'), new char[]{'я'});

        doubleCharTable.put(charsToInt('Y', 'O'), new char[]{'Ё'});
        doubleCharTable.put(charsToInt('Z', 'H'), new char[]{'Ж'});
        doubleCharTable.put(charsToInt('K', 'H'), new char[]{'Х'});
        doubleCharTable.put(charsToInt('T', 'S'), new char[]{'Ц'});
        doubleCharTable.put(charsToInt('C', 'H'), new char[]{'Ч'});
        doubleCharTable.put(charsToInt('S', 'H'), new char[]{'Ш'});
        doubleCharTable.put(charsToInt('Y', 'U'), new char[]{'Ю'});
        doubleCharTable.put(charsToInt('Y', 'A'), new char[]{'Я'});
        doubleCharTable.put(charsToInt('J', 'U'), new char[]{'Ю'});
        doubleCharTable.put(charsToInt('J', 'A'), new char[]{'Я'});

        return new TranslitDoubleCharTable(
            "translitEnRu",
            replacementTable,
            multiReplacementTable,
            doubleCharTable);
    }

    private static TranslitTable layoutRuEn() {
        char[] table = new char[1 << Character.SIZE];
        table['ё'] = '`';
        table['й'] = 'q';
        table['ц'] = 'w';
        table['у'] = 'e';
        table['к'] = 'r';
        table['е'] = 't';
        table['н'] = 'y';
        table['г'] = 'u';
        table['ш'] = 'i';
        table['щ'] = 'o';
        table['з'] = 'p';
        table['х'] = '[';
        table['ъ'] = ']';
        table['ф'] = 'a';
        table['ы'] = 's';
        table['в'] = 'd';
        table['а'] = 'f';
        table['п'] = 'g';
        table['р'] = 'h';
        table['о'] = 'j';
        table['л'] = 'k';
        table['д'] = 'l';
        table['ж'] = ';';
        table['э'] = '\'';
        table['я'] = 'z';
        table['ч'] = 'x';
        table['с'] = 'c';
        table['м'] = 'v';
        table['и'] = 'b';
        table['т'] = 'n';
        table['ь'] = 'm';
        table['б'] = ',';
        table['ю'] = '.';

        table['Ё'] = '~';
        table['Й'] = 'Q';
        table['Ц'] = 'W';
        table['У'] = 'E';
        table['К'] = 'R';
        table['Е'] = 'T';
        table['Н'] = 'Y';
        table['Г'] = 'U';
        table['Ш'] = 'I';
        table['Щ'] = 'O';
        table['З'] = 'P';
        table['Х'] = '{';
        table['Ъ'] = '}';
        table['Ф'] = 'A';
        table['Ы'] = 'S';
        table['В'] = 'D';
        table['А'] = 'F';
        table['П'] = 'G';
        table['Р'] = 'H';
        table['О'] = 'J';
        table['Л'] = 'K';
        table['Д'] = 'L';
        table['Ж'] = ':';
        table['Э'] = '"';
        table['Я'] = 'Z';
        table['Ч'] = 'X';
        table['С'] = 'C';
        table['М'] = 'V';
        table['И'] = 'B';
        table['Т'] = 'N';
        table['Ь'] = 'M';
        table['Б'] = '<';
        table['Ю'] = '>';

        //Do not convert "Парахин" to "Gfhf[by"
        //because '[' will break request to two tokens
        //wich will cause too many false positive results
        boolean[] exceptionsTable = new boolean[1 << Character.SIZE];
        exceptionsTable['ё'] = true;
        exceptionsTable['х'] = true;
        exceptionsTable['ъ'] = true;
        exceptionsTable['ж'] = true;
        exceptionsTable['э'] = true;
        exceptionsTable['б'] = true;
        exceptionsTable['ю'] = true;

        exceptionsTable['Ё'] = true;
        exceptionsTable['Х'] = true;
        exceptionsTable['Ъ'] = true;
        exceptionsTable['Ж'] = true;
        exceptionsTable['Э'] = true;
        exceptionsTable['Б'] = true;
        exceptionsTable['Ю'] = true;

        return new TranslitSingleCharTable(
            "layoutRuEn",
            table,
            exceptionsTable,
            new char[0][]);
    }

    private static TranslitTable layoutEnRu() {
        char[] table = new char[1 << Character.SIZE];
        table['`'] = 'ё';
        table['q'] = 'й';
        table['w'] = 'ц';
        table['e'] = 'у';
        table['r'] = 'к';
        table['t'] = 'е';
        table['y'] = 'н';
        table['u'] = 'г';
        table['i'] = 'ш';
        table['o'] = 'щ';
        table['p'] = 'з';
        table['['] = 'х';
        table[']'] = 'ъ';
        table['a'] = 'ф';
        table['s'] = 'ы';
        table['d'] = 'в';
        table['f'] = 'а';
        table['g'] = 'п';
        table['h'] = 'р';
        table['j'] = 'о';
        table['k'] = 'л';
        table['l'] = 'д';
        table[';'] = 'ж';
        table['\''] = 'э';
        table['z'] = 'я';
        table['x'] = 'ч';
        table['c'] = 'с';
        table['v'] = 'м';
        table['b'] = 'и';
        table['n'] = 'т';
        table['m'] = 'ь';
        table[','] = 'б';
        table['.'] = 'ю';

        table['~'] = 'Ё';
        table['Q'] = 'Й';
        table['W'] = 'Ц';
        table['E'] = 'У';
        table['R'] = 'К';
        table['T'] = 'Е';
        table['Y'] = 'Н';
        table['U'] = 'Г';
        table['I'] = 'Ш';
        table['O'] = 'Щ';
        table['P'] = 'З';
        table['['] = 'Х';
        table[']'] = 'Ъ';
        table['A'] = 'Ф';
        table['S'] = 'Ы';
        table['D'] = 'В';
        table['F'] = 'А';
        table['G'] = 'П';
        table['H'] = 'Р';
        table['J'] = 'О';
        table['K'] = 'Л';
        table['L'] = 'Д';
        table[':'] = 'Ж';
        table['"'] = 'Э';
        table['Z'] = 'Я';
        table['X'] = 'Ч';
        table['C'] = 'С';
        table['V'] = 'М';
        table['B'] = 'И';
        table['N'] = 'Т';
        table['M'] = 'Ь';
        table['<'] = 'Б';
        table['>'] = 'Ю';
        return new TranslitSingleCharTable(
            "layoutEnRu",
            table,
            new boolean[0],
            new char[0][]);
    }

    protected static int charsToInt(final char first, final char second) {
        return first | (second << Character.SIZE);
    }
}
