package ru.yandex.search.proxy.universal;

import java.util.function.Supplier;
import java.util.logging.Logger;

import org.apache.http.HttpException;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpRequest;
import org.apache.http.RequestLine;
import org.apache.http.client.protocol.HttpClientContext;

import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.EntityGenerator;
import ru.yandex.http.util.nio.client.AbstractAsyncClient;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.request.RequestInfo;
import ru.yandex.http.util.server.HttpServer;
import ru.yandex.parser.searchmap.User;
import ru.yandex.search.proxy.SearchProxyParams;
import ru.yandex.search.proxy.UpstreamContext;
import ru.yandex.search.request.config.SearchBackendRequestConfigBuilder;

public class BasicUniversalSearchProxyRequestContext
    implements UniversalSearchProxyRequestContext
{
    private final User user;
    private final Long minPos;
    private final AsyncClient client;
    private final Long failoverDelay;
    private final boolean localityShuffle;
    private final long lagTolerance;
    private final BasicAsyncRequestProducerGenerator producerGenerator;
    private final Supplier<? extends HttpClientContext>
        httpClientContextGenerator;
    private final Logger logger;

    // CSOFF: ParameterNumber
    public BasicUniversalSearchProxyRequestContext(
        final UniversalSearchProxy<?> proxy,
        final ProxySession session,
        final EntityGenerator requestBody,
        final boolean extractFailoverDelay)
        throws HttpException
    {
        logger = session.logger();
        user = proxy.extractUser(session);
        Long minPos = session.headers().getLong(
            YandexHeaders.X_MINIMAL_POSITION,
            null);
        if (minPos == null) {
            if (proxy.producerClient() == null) {
                this.minPos =
                    session.params().getLong(SearchProxyParams.MIN_POS);
            } else {
                this.minPos =
                    session.params().getLong(SearchProxyParams.MIN_POS, null);
            }
        } else {
            this.minPos = minPos;
        }

        HttpRequest request = session.request();
        RequestLine requestLine = request.getRequestLine();
        UpstreamContext upstream =
            proxy.upstreamFor(
                (RequestInfo) session.context().getAttribute(
                    HttpServer.REQUEST_INFO));

        client = upstream.client().adjust(session.context());

        SearchBackendRequestConfigBuilder requestConfig
            = new SearchBackendRequestConfigBuilder(
                session.params(),
            upstream.config().requestConfig());

        if (extractFailoverDelay) {
            Long failoverDelay = session.headers().getLongDuration(
                YandexHeaders.X_SEARCH_FAILOVER_DELAY,
                null);
            if (failoverDelay == null) {
                failoverDelay = requestConfig.failoverDelay();
            }
            this.failoverDelay = failoverDelay;
        } else {
            failoverDelay = null;
        }

        localityShuffle = requestConfig.localityShuffle();
        if (requestConfig.allowLaggingHosts()) {
            lagTolerance = Long.MAX_VALUE;
        } else {
            lagTolerance = 0L;
        }

        producerGenerator = new BasicAsyncRequestProducerGenerator(
            upstream.removePrefix(requestLine.getUri()),
            requestBody,
            requestLine.getMethod());
        producerGenerator.copyHeader(request, HttpHeaders.ACCEPT_CHARSET);
        producerGenerator.copyHeader(request, HttpHeaders.ACCEPT_ENCODING);

        httpClientContextGenerator =
            session.listener().createContextGeneratorFor(client);
    }
    // CSON: ParameterNumber

    @Override
    public User user() {
        return user;
    }

    @Override
    public Long minPos() {
        return minPos;
    }

    @Override
    public long lagTolerance() {
        return lagTolerance;
    }

    @Override
    public AbstractAsyncClient<AsyncClient> client() {
        return client;
    }

    @Override
    public Logger logger() {
        return logger;
    }

    public Long failoverDelay() {
        return failoverDelay;
    }

    public boolean localityShuffle() {
        return localityShuffle;
    }

    public BasicAsyncRequestProducerGenerator producerGenerator() {
        return producerGenerator;
    }

    public Supplier<? extends HttpClientContext> httpClientContextGenerator() {
        return httpClientContextGenerator;
    }
}

