package ru.yandex.search.proxy.universal;

import java.util.List;

import org.apache.http.HttpHost;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.client.producer.QueueHostInfo;
import ru.yandex.collection.LongPair;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.NotFoundException;

public class ParallelRequestCallback<T>
    implements FutureCallback<LongPair<T>>
{
    private enum Status {
        CONTINUE,
        ABORT,
        COMPLETE
    }

    private final UniversalSearchProxyRequestContext context;
    private final FutureCallback<? super T> callback;
    private Long minPos;
    private LongPair<T> bestResult = null;
    private Exception e = null;
    private Exception producerError = null;
    private boolean completed = false;
    private boolean allGood = true;
    private int onAir;

    public ParallelRequestCallback(
        final UniversalSearchProxyRequestContext context,
        final FutureCallback<? super T> callback,
        final List<HttpHost> hosts,
        final Long minPos)
    {
        this.context = context;
        this.callback = callback;
        this.minPos = minPos;
        onAir = hosts.size();
    }

    private Status checkStatus() {
        if (onAir == 0) {
            if (allGood) {
                // All backends responded, bestResult contains the result
                return Status.COMPLETE;
            } else if (bestResult == null) {
                // No successful responses at all
                return Status.ABORT;
            } else if (minPos == null) {
                // Some backends completed, some not and we don't know position
                if (producerError == null) {
                    // Wait for producer to provide position
                    return Status.CONTINUE;
                } else {
                    // Producer failed, check if bestResult is acceptable
                    if (context.lagTolerance() == Long.MAX_VALUE) {
                        // Lagging hosts allowed
                        return Status.COMPLETE;
                    } else {
                        return Status.ABORT;
                    }
                }
            } else {
                long lag = minPos.longValue() - bestResult.first();
                if (lag <= context.lagTolerance()) {
                    return Status.COMPLETE;
                } else {
                    return Status.ABORT;
                }
            }
        } else {
            if (allGood) {
                // No backend failures yet
                if (minPos == null) {
                    // Probably producer already failed but we still can return
                    // actual result if all backends respond
                    return Status.CONTINUE;
                } else if (bestResult == null) {
                    // Not a single backend responded yet, let's wait
                    return Status.CONTINUE;
                } else if (bestResult.first() >= minPos.longValue()) {
                    // Wow! Such actual!
                    return Status.COMPLETE;
                } else {
                    // We don't have response from actual backend yet
                    return Status.CONTINUE;
                }
            } else {
                // Some hosts failed. Producer is our only hope now
                if (minPos == null) {
                    // We don't have actual position, check if producer failed
                    if (producerError == null) {
                        // Let's wait producer to respond
                        return Status.CONTINUE;
                    } else {
                        // Producer failed, check if lagging hosts allowed
                        if (context.lagTolerance() == Long.MAX_VALUE) {
                            // Let's wait for all backends to respond, so we
                            // can select most actual response
                            return Status.CONTINUE;
                        } else {
                            return Status.ABORT;
                        }
                    }
                } else {
                    // Check if we already have acceptable response
                    if (bestResult == null) {
                        // Nope, nothing. Wait for any backend to respond
                        return Status.CONTINUE;
                    } else if (bestResult.first() >= minPos.longValue()) {
                        // Got actual response
                        return Status.COMPLETE;
                    } else {
                        // Wait for another backend to respond
                        return Status.CONTINUE;
                    }
                }
            }
        }
    }

    @Override
    public void cancelled() {
        Status status;
        synchronized (this) {
            if (completed) {
                return;
            }
            allGood = false;
            --onAir;
            status = checkStatus();
            if (status == Status.CONTINUE) {
                return;
            }
            completed = true;
        }
        if (status == Status.ABORT) {
            callback.cancelled();
        } else {
            callback.completed(bestResult.second());
        }
    }

    @Override
    public void failed(final Exception e) {
        Status status;
        synchronized (this) {
            if (completed) {
                return;
            }
            if (this.e == null) {
                this.e = e;
            } else {
                this.e.addSuppressed(e);
            }
            allGood = false;
            --onAir;
            status = checkStatus();
            if (status == Status.CONTINUE) {
                return;
            }
            completed = true;
        }
        if (status == Status.ABORT) {
            callback.failed(e);
        } else {
            callback.completed(bestResult.second());
        }
    }

    @Override
    public void completed(final LongPair<T> result) {
        Status status;
        synchronized (this) {
            if (completed) {
                return;
            }
            --onAir;
            if (bestResult == null || bestResult.first() < result.first()) {
                bestResult = result;
            }
            status = checkStatus();
            if (status == Status.CONTINUE) {
                return;
            }
            completed = true;
        }

        if (status == Status.ABORT) {
            callback.failed(e);
        } else {
            callback.completed(bestResult.second());
        }
    }

    public ProducerCallback producerCallback() {
        return new ProducerCallback();
    }

    public class ProducerCallback
        extends AbstractFilterFutureCallback<List<QueueHostInfo>, T>
    {
        ProducerCallback() {
            super(ParallelRequestCallback.this.callback);
        }

        @Override
        public void completed(final List<QueueHostInfo> infos) {
            if (infos.isEmpty()) {
                synchronized (ParallelRequestCallback.this) {
                    if (completed) {
                        return;
                    }
                    completed = true;
                }
                callback.failed(
                    new NotFoundException(
                        "No hosts found for user " + context.user()));
            } else {
                Long pos = infos.get(0).queueId();
                Status status;
                synchronized (ParallelRequestCallback.this) {
                    if (completed) {
                        return;
                    }
                    minPos = pos;
                    status = checkStatus();
                    if (status == Status.CONTINUE) {
                        return;
                    }
                    completed = true;
                }

                if (status == Status.ABORT) {
                    callback.failed(e);
                } else {
                    callback.completed(bestResult.second());
                }
            }
        }

        @Override
        public void failed(final Exception e) {
            Status status;
            synchronized (ParallelRequestCallback.this) {
                if (completed) {
                    return;
                }
                producerError = e;
                status = checkStatus();
                if (status == Status.CONTINUE) {
                    return;
                }
                completed = true;
            }

            if (status == Status.ABORT) {
                callback.failed(producerError);
            } else {
                callback.completed(bestResult.second());
            }
        }
    }
}

