package ru.yandex.search.proxy.universal;

import java.util.ArrayList;
import java.util.List;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.http.HttpHost;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.nio.protocol.HttpAsyncRequestProducer;

import ru.yandex.client.producer.QueueHostInfo;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.CancellationSubscriber;
import ru.yandex.http.util.HttpHostAppender;
import ru.yandex.http.util.NotFoundException;
import ru.yandex.http.util.nio.HttpAsyncResponseConsumerFactory;
import ru.yandex.util.string.StringUtils;

public class SequentialRequestCallback<T>
    extends AbstractFilterFutureCallback<List<QueueHostInfo>, T>
{
    private final CancellationSubscriber cancellationSubscriber;
    private final UniversalSearchProxyRequestContext context;
    private final Function
        <? super HttpHost, ? extends HttpAsyncRequestProducer>
        producerGenerator;
    private final Long failoverDelay;
    private final boolean localityShuffle;
    private final HttpAsyncResponseConsumerFactory<T> consumerFactory;
    private final Supplier<? extends HttpClientContext> contextGenerator;

    public SequentialRequestCallback(
        final CancellationSubscriber cancellationSubscriber,
        final UniversalSearchProxyRequestContext context,
        final Function<? super HttpHost, ? extends HttpAsyncRequestProducer>
        producerGenerator,
        final Long failoverDelay,
        final boolean localityShuffle,
        final HttpAsyncResponseConsumerFactory<T> consumerFactory,
        final Supplier<? extends HttpClientContext> contextGenerator,
        final FutureCallback<? super T> callback)
    {
        super(callback);
        this.cancellationSubscriber = cancellationSubscriber;
        this.context = context;
        this.producerGenerator = producerGenerator;
        this.failoverDelay = failoverDelay;
        this.localityShuffle = localityShuffle;
        this.consumerFactory = consumerFactory;
        this.contextGenerator = contextGenerator;
    }

    @Override
    public void completed(final List<QueueHostInfo> infos) {
        int size = infos.size();
        List<HttpHost> hosts = new ArrayList<>(size);
        Long minPos = context.minPos();
        if (minPos == null && size > 0) {
            minPos = infos.get(0).queueId();
        }
        int actualCount = 0;
        for (int i = 0; i < size; ++i) {
            QueueHostInfo info = infos.get(i);
            long lag = minPos.longValue() - info.queueId();
            if (lag <= 0L) {
                hosts.add(info.host());
                actualCount++;
            } else if (lag <= context.lagTolerance()) {
                hosts.add(info.host());
            } else {
                break;
            }
        }
        Logger logger = context.logger();
        if (logger.isLoggable(Level.FINE)) {
            context.logger().fine(
                new String(
                    StringUtils.joinStringBuilderables(
                        new StringBuilder("producer-hostlist: "),
                        infos,
                        ',')));
        }
        if (hosts.isEmpty()) {
            failed(
                new NotFoundException(
                    "No hosts found for user " + context.user()
                    + " for position " + minPos
                    + " among producer hosts " + infos));
        } else {
            if (hosts.size() == 1) {
                cancellationSubscriber.subscribeForCancellation(
                    context.client().execute(
                        hosts.get(0),
                        producerGenerator,
                        consumerFactory,
                        contextGenerator,
                        callback));
            } else {
                List<HttpHost> actualHosts = hosts.subList(0, actualCount);
                if (localityShuffle) {
                    UniversalSearchProxy.localityShuffle(actualHosts);
                } else {
                    UniversalSearchProxy.shuffle(actualHosts, context.user());
                }
                // actualHosts is just a view over host and they are
                // already properly shuflled
                // non-actual hosts are sorted by lag, so no shuffle needed
                // if lagging hosts not allowed, hosts will contain only
                // actual hosts
                if (logger.isLoggable(Level.FINE)) {
                    context.logger().fine(
                        new String(
                            StringUtils.join(
                                new StringBuilder("shuffled-list: "),
                                hosts,
                                HttpHostAppender.INSTANCE,
                                ',')));
                }
                if (failoverDelay == null) {
                    cancellationSubscriber.subscribeForCancellation(
                        context.client().execute(
                            hosts,
                            producerGenerator,
                            consumerFactory,
                            contextGenerator,
                            callback));
                } else {
                    cancellationSubscriber.subscribeForCancellation(
                        context.client().executeWithDelay(
                            hosts,
                            producerGenerator,
                            failoverDelay,
                            consumerFactory,
                            contextGenerator,
                            callback));
                }
            }
        }
    }
}

