package ru.yandex.search.proxy.universal;

import java.io.IOException;
import java.util.List;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.logging.Logger;

import org.apache.http.HttpHost;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.nio.protocol.HttpAsyncRequestProducer;
import org.apache.http.protocol.HttpContext;

import ru.yandex.client.producer.ImmutableProducerClientConfig;
import ru.yandex.client.producer.ProducerClient;
import ru.yandex.collection.Pattern;
import ru.yandex.collection.PatternMap;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.CancellationSubscriber;
import ru.yandex.http.util.IdempotentFutureCallback;
import ru.yandex.http.util.nio.HttpAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.ZooQueueIdAsyncConsumerFactory;
import ru.yandex.http.util.nio.client.AbstractAsyncClient;
import ru.yandex.http.util.nio.client.RequestsListener;
import ru.yandex.http.util.request.RequestInfo;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.parser.searchmap.User;
import ru.yandex.search.proxy.SearchProxy;
import ru.yandex.search.request.SearchBackendRequestType;
import ru.yandex.search.request.config.SearchBackendRequestConfig;

public class UniversalSearchProxy
    <C extends ImmutableUniversalSearchProxyConfig>
    extends SearchProxy<C>
{
    public static final String SEQUENTIAL = "/sequential";
    public static final String PARALLEL = "/parallel";

    private final ProducerClient producerClient;
    private final HttpHost producerHost;

    public UniversalSearchProxy(final C config) throws IOException {
        super(config);

        ImmutableProducerClientConfig producerClientConfig =
            config.producerClientConfig();
        if (producerClientConfig == null) {
            producerClient = null;
            producerHost = null;
        } else {
            producerClient =
                registerClient(
                    "Producer",
                    new ProducerClient(
                        reactor,
                        config.producerClientConfig(),
                        searchMap()),
                    config.producerClientConfig());
            producerHost = producerClientConfig.host();
            register(
                new Pattern<>(SEQUENTIAL + '/', true),
                new SequentialRequestHandler(this));
        }
        register(
            new Pattern<>(PARALLEL + '/', true),
            new ParallelRequestHandler(this));
    }

    public ProducerClient producerClient() {
        return producerClient;
    }

    public HttpHost producerHost() {
        return producerHost;
    }

    public <T> void sequentialRequest(
        final ProxySession session,
        final UniversalSearchProxyRequestContext requestContext,
        final Function<? super HttpHost, ? extends HttpAsyncRequestProducer>
        producerGenerator,
        final Long failoverDelay,
        final boolean localityShuffle,
        final HttpAsyncResponseConsumerFactory<T> consumerFactory,
        final Supplier<? extends HttpClientContext> contextGenerator,
        final FutureCallback<? super T> callback)
    {
        sequentialRequest(
            session,
            session.listener(),
            session.context(),
            requestContext,
            producerGenerator,
            failoverDelay,
            localityShuffle,
            consumerFactory,
            contextGenerator,
            callback);
    }

    public <T> void sequentialRequest(
        final CancellationSubscriber cancellationSubscriber,
        final RequestsListener requestsListener,
        final HttpContext httpContext,
        final UniversalSearchProxyRequestContext requestContext,
        final Function<? super HttpHost, ? extends HttpAsyncRequestProducer>
        producerGenerator,
        final Long failoverDelay,
        final boolean localityShuffle,
        final HttpAsyncResponseConsumerFactory<T> consumerFactory,
        final Supplier<? extends HttpClientContext> contextGenerator,
        final FutureCallback<? super T> callback)
    {
        ProducerClient client = producerClient.adjust(httpContext);
        sequentialRequest(
            cancellationSubscriber,
            requestContext,
            producerGenerator,
            failoverDelay,
            localityShuffle,
            consumerFactory,
            contextGenerator,
            callback,
            client,
            requestsListener.createContextGeneratorFor(client));
    }

    public <T> void sequentialRequest(
        final CancellationSubscriber cancellationSubscriber,
        final UniversalSearchProxyRequestContext requestContext,
        final Function<? super HttpHost, ? extends HttpAsyncRequestProducer>
        producerGenerator,
        final Long failoverDelay,
        final boolean localityShuffle,
        final HttpAsyncResponseConsumerFactory<T> consumerFactory,
        final Supplier<? extends HttpClientContext> contextGenerator,
        final FutureCallback<? super T> callback,
        final ProducerClient producerClient,
        final Supplier<? extends HttpClientContext> producerContextGenerator)
    {
        cancellationSubscriber.subscribeForCancellation(
            producerClient.executeAllWithInfo(
                requestContext.user(),
                producerContextGenerator,
                new SequentialRequestCallback<T>(
                    cancellationSubscriber,
                    requestContext,
                    producerGenerator,
                    failoverDelay,
                    localityShuffle,
                    consumerFactory,
                    contextGenerator,
                    callback)));
    }

    public <T> void parallelRequest(
        final ProxySession session,
        final UniversalSearchProxyRequestContext requestContext,
        final Function<? super HttpHost, ? extends HttpAsyncRequestProducer>
            producerGenerator,
        final HttpAsyncResponseConsumerFactory<T> consumerFactory,
        final Supplier<? extends HttpClientContext> contextGenerator,
        final FutureCallback<? super T> callback)
    {
        parallelRequest(
            session,
            session.listener(),
            session.context(),
            requestContext,
            producerGenerator,
            consumerFactory,
            contextGenerator,
            callback);
    }

    public <T> void parallelRequest(
        final CancellationSubscriber cancellationSubscriber,
        final RequestsListener requestsListener,
        final HttpContext httpContext,
        final UniversalSearchProxyRequestContext requestContext,
        final Function<? super HttpHost, ? extends HttpAsyncRequestProducer>
            producerGenerator,
        final HttpAsyncResponseConsumerFactory<T> consumerFactory,
        final Supplier<? extends HttpClientContext> contextGenerator,
        final FutureCallback<? super T> callback)
    {
        List<HttpHost> hosts = searchMap().searchHosts(requestContext.user());
        parallelRequest(
            cancellationSubscriber,
            requestsListener,
            httpContext,
            hosts,
            requestContext,
            producerGenerator,
            consumerFactory,
            contextGenerator,
            new ParallelRequestCallback<>(
                requestContext,
                callback,
                hosts,
                requestContext.minPos()));
    }

    public <T> void parallelRequest(
        final ProxySession session,
        final List<HttpHost> hosts,
        final UniversalSearchProxyRequestContext requestContext,
        final Function<? super HttpHost, ? extends HttpAsyncRequestProducer>
        producerGenerator,
        final HttpAsyncResponseConsumerFactory<T> consumerFactory,
        final Supplier<? extends HttpClientContext> contextGenerator,
        final ParallelRequestCallback<T> parallelCallback)
    {
        parallelRequest(
            session,
            session.listener(),
            session.context(),
            hosts,
            requestContext,
            producerGenerator,
            consumerFactory,
            contextGenerator,
            parallelCallback);
    }

    @SuppressWarnings("FutureReturnValueIgnored")
    public <T> void parallelRequest(
        final CancellationSubscriber cancellationSubscriber,
        final RequestsListener requestsListener,
        final HttpContext httpContext,
        final List<HttpHost> hosts,
        final UniversalSearchProxyRequestContext requestContext,
        final Function<? super HttpHost, ? extends HttpAsyncRequestProducer>
        producerGenerator,
        final HttpAsyncResponseConsumerFactory<T> consumerFactory,
        final Supplier<? extends HttpClientContext> contextGenerator,
        final ParallelRequestCallback<T> parallelCallback)
    {
        Long minPos = requestContext.minPos();
        if (minPos == null) {
            ProducerClient client = producerClient.adjust(httpContext);
            client.executeAllWithInfo(
                requestContext.user(),
                requestsListener.createContextGeneratorFor(client),
                new IdempotentFutureCallback<>(
                    parallelCallback.producerCallback()));
        }
        ZooQueueIdAsyncConsumerFactory<T> zooQueueIdConsumerFactory =
            new ZooQueueIdAsyncConsumerFactory<>(consumerFactory);
        for (HttpHost host: hosts) {
            cancellationSubscriber.subscribeForCancellation(
                requestContext.client().execute(
                    host,
                    producerGenerator,
                    zooQueueIdConsumerFactory,
                    contextGenerator,
                    new IdempotentFutureCallback<>(parallelCallback)));
        }
    }

    public void searchRequest(
        final User user,
        final ProxySession session,
        final String requestTag,
        final Function<? super HttpHost, ? extends HttpAsyncRequestProducer>
            producerGenerator,
        final FutureCallback<JsonObject> callback)
    {
        this.searchRequest(
            user,
            session,
            requestTag,
            producerGenerator,
            JsonAsyncTypesafeDomConsumerFactory.OK,
            session.listener().createContextGeneratorFor(searchClient()),
            callback);
    }

    public <T> void searchRequest(
        final User user,
        final ProxySession session,
        final Function<? super HttpHost, ? extends HttpAsyncRequestProducer>
            producerGenerator,
        final HttpAsyncResponseConsumerFactory<T> consumerFactory,
        final Supplier<? extends HttpClientContext> contextGenerator,
        final FutureCallback<? super T> callback)
    {
        this.searchRequest(
            user,
            session,
            PatternMap.ASTERISK,
            producerGenerator,
            consumerFactory,
            contextGenerator,
            callback);
    }

    public <T> void searchRequest(
        final User user,
        final ProxySession session,
        final String requestName,
        final Function<? super HttpHost, ? extends HttpAsyncRequestProducer>
            producerGenerator,
        final HttpAsyncResponseConsumerFactory<T> consumerFactory,
        final Supplier<? extends HttpClientContext> contextGenerator,
        final FutureCallback<? super T> callback)
    {
        RequestInfo info = new RequestInfo(session.request());
        SearchBackendRequestConfig requestConfig =
            config.searchBackendRequestsConfig().configs().get(info).get(requestName);
        searchRequest(
            user,
            requestConfig,
            searchClient.adjust(session.context()),
            session,
            session.listener(),
            session.context(),
            session.logger(),
            producerGenerator,
            consumerFactory,
            contextGenerator,
            callback);
    }


    public <T> void searchRequest(
        final User user,
        final SearchBackendRequestConfig requestConfig,
        final AbstractAsyncClient<?> client,
        final CancellationSubscriber cancellationSubscriber,
        final RequestsListener requestsListener,
        final HttpContext httpContext,
        final Logger logger,
        final Function<? super HttpHost, ? extends HttpAsyncRequestProducer>
            producerGenerator,
        final HttpAsyncResponseConsumerFactory<T> consumerFactory,
        final Supplier<? extends HttpClientContext> contextGenerator,
        final FutureCallback<? super T> callback)
    {
        if (requestConfig.type() == SearchBackendRequestType.SEQUENTIAL) {
            sequentialRequest(
                cancellationSubscriber,
                requestsListener,
                httpContext,
                new SearchRequestContext(requestConfig, user, logger, client),
                producerGenerator,
                requestConfig.failoverDelay(),
                requestConfig.localityShuffle(),
                consumerFactory,
                contextGenerator,
                callback);
        } else {
            parallelRequest(
                cancellationSubscriber,
                requestsListener,
                httpContext,
                new SearchRequestContext(requestConfig, user, logger, client),
                producerGenerator,
                consumerFactory,
                contextGenerator,
                callback);
        }
    }

    private final static class SearchRequestContext implements UniversalSearchProxyRequestContext {
        private final SearchBackendRequestConfig config;
        private final User user;
        private final Logger logger;
        private final AbstractAsyncClient<?> client;

        public SearchRequestContext(
            final SearchBackendRequestConfig config,
            final User user,
            final Logger logger,
            final AbstractAsyncClient<?> client)
        {
            this.config = config;
            this.user = user;
            this.logger = logger;
            this.client = client;
        }

        @Override
        public User user() {
            return user;
        }

        @Override
        public Long minPos() {
            return config.minPos();
        }

        @Override
        public AbstractAsyncClient<?> client() {
            return client;
        }

        @Override
        public Logger logger() {
            return logger;
        }

        @Override
        public long lagTolerance() {
            if (config.allowLaggingHosts()) {
                return Long.MAX_VALUE;
            } else {
                return 0L;
            }
        }
    }
}

