package ru.yandex.mail.search.web;

import java.io.IOException;
import java.util.List;
import java.util.function.Supplier;
import java.util.logging.Logger;

import org.apache.http.HttpException;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpHost;
import org.apache.http.HttpRequest;
import org.apache.http.RequestLine;
import org.apache.http.client.protocol.HttpClientContext;

import ru.yandex.client.producer.ProducerClient;
import ru.yandex.http.proxy.HttpEntitySendingCallback;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.NByteArrayEntityAsyncConsumerFactory;
import ru.yandex.http.util.nio.client.AbstractAsyncClient;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.request.RequestInfo;
import ru.yandex.http.util.server.HttpServer;
import ru.yandex.parser.searchmap.User;
import ru.yandex.search.proxy.SearchProxyParams;
import ru.yandex.search.proxy.UpstreamContext;
import ru.yandex.search.proxy.universal.SequentialRequestCallback;
import ru.yandex.search.proxy.universal.UniversalSearchProxyRequestContext;

public class BackendSearchHandler implements ProxyRequestHandler {
    public static final String URI_PREFIX = "/backend";

    private final DefaultPsProject project;

    public BackendSearchHandler(final DefaultPsProject project) {
        this.project = project;
    }

    @Override
    public void handle(
        final ProxySession session)
        throws HttpException, IOException
    {
        WebtoolsSearchContext requestContext =
            new WebtoolsSearchContext(session, project);

        if (project.producerClient() != null) {
            ProducerClient client =
                project.producerClient().adjust(session.context());
            client.executeAllWithInfo(
                requestContext.user(),
                session.listener().createContextGeneratorFor(client),
                new SequentialRequestCallback<>(
                    session,
                    requestContext,
                    requestContext.producerGenerator(),
                    requestContext.failoverDelay(),
                    requestContext.localityShuffle(),
                    NByteArrayEntityAsyncConsumerFactory.OK,
                    requestContext.httpClientContextGenerator(),
                    new HttpEntitySendingCallback(session)));
        } else {
            List<HttpHost> hosts =
                project.searchmap().searchHosts(requestContext.user());

            requestContext.client().execute(
                hosts,
                requestContext.producerGenerator(),
                NByteArrayEntityAsyncConsumerFactory.OK,
                requestContext.httpClientContextGenerator(),
                new HttpEntitySendingCallback(session));
        }
    }

    private static final class WebtoolsSearchContext
        implements UniversalSearchProxyRequestContext {
        private final User user;
        private final Long minPos;
        private final AsyncClient client;
        private final Long failoverDelay;
        private final boolean localityShuffle;
        private final boolean allowLaggingHosts;
        private final BasicAsyncRequestProducerGenerator producerGenerator;
        private final Supplier<? extends HttpClientContext>
            httpClientContextGenerator;
        private final Logger logger;

        private WebtoolsSearchContext(
            final ProxySession session,
            final DefaultPsProject project)
            throws HttpException
        {
            logger = session.logger();
            user = project.extractUser(session);
            Long minPos = session.headers().getLong(
                YandexHeaders.X_MINIMAL_POSITION,
                null);
            if (minPos == null) {
                if (project.producerClient() == null) {
                    this.minPos =
                        session.params().getLong(SearchProxyParams.MIN_POS, -1L);
                } else {
                    this.minPos =
                        session.params().getLong(SearchProxyParams.MIN_POS, -1L);
                }
            } else {
                this.minPos = minPos;
            }

            HttpRequest request = session.request();
            RequestLine requestLine = request.getRequestLine();
            UpstreamContext upstream =
                project.webApi().upstreamFor(
                    (RequestInfo) session.context().getAttribute(
                        HttpServer.REQUEST_INFO));

            client = upstream.client().adjust(session.context());

            Long failoverDelay = session.headers().getLongDuration(
                YandexHeaders.X_SEARCH_FAILOVER_DELAY,
                null);
            if (failoverDelay == null) {
                failoverDelay = session.params().getLongDuration(
                    SearchProxyParams.FAILOVER_DELAY,
                    upstream.config().requestConfig().failoverDelay());
            }
            this.failoverDelay = failoverDelay;

            localityShuffle = upstream.config().requestConfig().localityShuffle();

            allowLaggingHosts = upstream.config().requestConfig().allowLaggingHosts();

            String uri = upstream.removePrefix(requestLine.getUri());
            uri = uri.substring(URI_PREFIX.length());
            producerGenerator = new BasicAsyncRequestProducerGenerator(uri);
            producerGenerator.copyHeader(request, HttpHeaders.ACCEPT_CHARSET);
            producerGenerator.copyHeader(request, HttpHeaders.ACCEPT_ENCODING);

            httpClientContextGenerator =
                session.listener().createContextGeneratorFor(client);
        }

        @Override
        public User user() {
            return user;
        }

        @Override
        public Long minPos() {
            return minPos;
        }

        @Override
        public AbstractAsyncClient<AsyncClient> client() {
            return client;
        }

        @Override
        public Logger logger() {
            return logger;
        }

        public Long failoverDelay() {
            return failoverDelay;
        }

        public boolean localityShuffle() {
            return localityShuffle;
        }

        @Override
        public long lagTolerance() {
            return allowLaggingHosts ? Long.MAX_VALUE : 0L;
        }

        public BasicAsyncRequestProducerGenerator producerGenerator() {
            return producerGenerator;
        }

        public Supplier<? extends HttpClientContext> httpClientContextGenerator() {
            return httpClientContextGenerator;
        }
    }
}
