package ru.yandex.mail.search.web;

import java.io.IOException;
import java.text.ParseException;

import org.apache.http.HttpException;

import ru.yandex.client.producer.ProducerClient;
import ru.yandex.collection.Pattern;
import ru.yandex.http.config.HttpTargetConfigBuilder;
import ru.yandex.http.config.ImmutableRetriesConfig;
import ru.yandex.http.config.RetriesConfigBuilder;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.request.RequestHandlerMapper;
import ru.yandex.json.writer.JsonWriterBase;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.mail.search.web.config.ImmutablePsProjectConfig;
import ru.yandex.mail.search.web.drop.DropLuceneHostHandler;
import ru.yandex.mail.search.web.drop.DropLuceneShardHandler;
import ru.yandex.mail.search.web.health.HealthCheckService;
import ru.yandex.mail.search.web.health.HealthInfoHandler;
import ru.yandex.mail.search.web.health.base.PsProjectType;
import ru.yandex.mail.search.web.searchmap.ListServicesHandler;
import ru.yandex.mail.search.web.searchmap.ListShardsWithFilterHandler;
import ru.yandex.mail.search.web.searchmap.SearchmapHelper;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.searchmap.SearchMap;
import ru.yandex.parser.searchmap.ShardNumberValidator;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.search.mop.manager.MopManager;
import ru.yandex.search.prefix.Prefix;
import ru.yandex.search.prefix.ShardPrefix;
import ru.yandex.search.proxy.SearchProxyParams;

public class DefaultPsProject implements WebtoolsProject {
    private static final int DROP_SHARD_TIMEOUT = 55000;
    private static final int RESET_TIMEOUT = 30000;
    private final ImmutablePsProjectConfig config;

    protected final WebApi webApi;
    protected final SearchMap searchmap;
    protected final AsyncClient queueClient;
    protected final AsyncClient dropShardClient;
    protected final AsyncClient resetShardClient;
    protected final PrefixedLogger logger;
    protected final PsProjectType project;
    protected final HealthCheckService healthCheckService;
    protected final ProducerClient producerClient;
    protected final MopManager mopManager;
    protected final SearchmapHelper helper;

    public DefaultPsProject(
        final WebApi webApi,
        final PsProjectType project,
        final ImmutablePsProjectConfig config)
        throws IOException
    {
        this.webApi = webApi;
        this.config = config;
        this.project = project;

        this.logger = webApi.logger().addPrefix(config.projectId());
        try {
            if (config.mopConfig() != null) {
                mopManager = new MopManager(config.mopConfig(), webApi.config().dnsConfig(), this.logger);
                searchmap = mopManager.searchMapWrapper();
            } else {
                searchmap = config.searchMapConfig().build();
                mopManager = null;
            }

            if (webApi.config().producerClientConfig() != null) {
                producerClient =
                    webApi.registerClient(
                        config.projectId() + "-ProducerClient",
                        new ProducerClient(
                            webApi.reactor(),
                            webApi.config().producerClientConfig(),
                            searchmap),
                        webApi.config().producerClientConfig());
            } else {
                producerClient = null;
            }

            queueClient = webApi.client(
                config.projectId() + "-QueueClient",
                new HttpTargetConfigBuilder().connections(2).build());

            dropShardClient = webApi.client(
                config.projectId() + "-DropShardClient",
                new HttpTargetConfigBuilder().connections(2).timeout(
                    DROP_SHARD_TIMEOUT).build());
            ImmutableRetriesConfig retriesConfig =
                new RetriesConfigBuilder().count(5).build();

            resetShardClient = webApi.client(
                config.projectId() + "-ResetShardClient",
                new HttpTargetConfigBuilder()
                    .connections(10)
                    .timeout(RESET_TIMEOUT)
                    .httpRetries(retriesConfig)
                    .ioRetries(retriesConfig)
                    .build());
        } catch (ParseException | ConfigException pe) {
            throw new IOException(pe);
        }

        if (mopManager != null) {
            helper = new SearchmapHelper(logger, mopManager::searchMapWrapper, config.totalShardsCount());
        } else {
            helper = new SearchmapHelper(logger, () -> searchmap, config.totalShardsCount());
        }

        if (config.metrics().size() > 0) {
            healthCheckService = new HealthCheckService(this, helper, config);
        } else {
            healthCheckService = null;
        }

        webApi.register(
            config,
            "/update_position",
            new UpdateLucenePositionHandler(this));

        webApi.register(
            config,
            "/searchmap/services",
            new ListServicesHandler(helper),
            RequestHandlerMapper.GET);

        webApi.register(
            config,
            "/queue/get",
            new QueueGetHandler(this),
            RequestHandlerMapper.GET);

        webApi.register(
            config,
            "/searchmap/shards/filter",
            new ListShardsWithFilterHandler(this, helper),
            RequestHandlerMapper.GET);

        webApi.register(
            config,
            "/copy/job/create",
            new ListShardsWithFilterHandler(this, helper),
            RequestHandlerMapper.GET);

        if (healthCheckService != null) {
            webApi.register(
                config,
                "/health/info",
                healthInfoHandler(),
                RequestHandlerMapper.GET);
            webApi.register(
                config,
                "/health/shard/drop",
                dropShardHandler(),
                RequestHandlerMapper.GET
            );

            webApi.register(
                config,
                "/health/host/drop",
                dropHostHandler(),
                RequestHandlerMapper.GET
            );
        }

        webApi.register(
            new Pattern<>(
                '/' + config.projectId() + BackendSearchHandler.URI_PREFIX,
                true),
            new BackendSearchHandler(this));
    }

    protected ProxyRequestHandler healthInfoHandler() {
        return new HealthInfoHandler(healthCheckService);
    }

    protected ProxyRequestHandler dropShardHandler() {
        return new DropLuceneShardHandler(healthCheckService);
    }

    protected ProxyRequestHandler dropHostHandler() {
        return new DropLuceneHostHandler(healthCheckService);
    }

    @Override
    public void start() {
        if (healthCheckService != null) {
            healthCheckService.start();
        }

        if (mopManager != null) {
            mopManager.start();
        }
    }

    @Override
    public void close() throws IOException {
        if (healthCheckService != null) {
            healthCheckService.close();
        }
    }

    @Override
    public SearchMap searchmap() {
        return searchmap;
    }

    @Override
    public String defaultService(final String prefix) {
        return null;
    }

    @Override
    public AsyncClient searchClient() {
        return null;
    }

    public AsyncClient queueClient() {
        return queueClient;
    }

    @Override
    public PrefixedLogger logger() {
        return logger;
    }

    public WebApi webApi() {
        return webApi;
    }

    public PsProjectType project() {
        return project;
    }

    public ImmutablePsProjectConfig config() {
        return config;
    }

    @Override
    public String defaultService() {
        return config.defaultService();
    }

    @Override
    public String hostYasmPanel(final String hostname) {
        String name = hostname;

        if (!name.endsWith(".search.yandex.net")) {
            String[] split = name.split("-");
            if (split.length > 2) {
                name = split[0] + '-' + split[1] + ".search.yandex.net";
            }
        }
        return
            "https://yasm.yandex-team.ru/template/panel/"
                + config.yasmLuceneTemplate()
                + "/prj=" + config.yasmProject()
                + ";ctype=prod,prestable;hosts=" + name + '/';
    }

    @Override
    public void writeValue(final JsonWriterBase writer) throws IOException {
        writer.startObject();
        writer.key("id");
        writer.value(config.projectId());
        writer.key("name");
        writer.value(config.name());
        writer.key("description");
        writer.value(config.description());
        writer.endObject();
    }

    public AsyncClient dropShardClient() {
        return dropShardClient;
    }

    public AsyncClient resetShardClient() {
        return resetShardClient;
    }

    public ProducerClient producerClient() {
        return producerClient;
    }

    protected User extractUser(final ProxySession session)
        throws HttpException
    {
        String service = session.headers().get(
            YandexHeaders.SERVICE,
            null,
            NonEmptyValidator.INSTANCE);
        Prefix prefix;
        if (service == null) {
            service = session.params().getString(
                SearchProxyParams.SERVICE,
                defaultService());
            Long shard = session.params().get(
                SearchProxyParams.SHARD,
                null,
                ShardNumberValidator.INSTANCE);
            if (shard == null) {
                prefix = session.params().get(
                    SearchProxyParams.PREFIX,
                    searchmap.prefixType(service));
            } else {
                prefix = new ShardPrefix(shard);
            }
        } else {
            Long shard = session.headers().get(
                YandexHeaders.ZOO_SHARD_ID,
                null,
                ShardNumberValidator.INSTANCE);
            if (shard == null) {
                prefix = session.headers().get(
                    YandexHeaders.X_SEARCH_PREFIX,
                    searchmap.prefixType(service));
            } else {
                prefix = new ShardPrefix(shard);
            }
        }
        return new User(service, prefix);
    }
}
