package ru.yandex.mail.search.web;

import java.io.IOException;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.apache.http.HttpException;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.blackbox.BlackboxUserIdType;
import ru.yandex.blackbox.BlackboxUserinfo;
import ru.yandex.blackbox.BlackboxUserinfoRequest;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;

public class LinkedAccountsHandler
    implements ProxyRequestHandler
{
    private static final String LOGIN = "login";
    private final WebApi webApi;

    public LinkedAccountsHandler(final WebApi webApi) {
        this.webApi = webApi;
    }

    @Override
    public void handle(
        final ProxySession session)
        throws HttpException, IOException
    {
        String corpLogin = session.params().getString(LOGIN);
        AsyncClient staffClient =
            webApi.staffClient().adjust(session.context());
        BasicAsyncRequestProducerGenerator generator =
            new BasicAsyncRequestProducerGenerator(
                "/v3/persons?login=" + corpLogin + "&_one=1");
        generator.addHeader(
            HttpHeaders.AUTHORIZATION,
            webApi.config().staffToken());
        staffClient.execute(
            webApi.config().staff().host(),
            generator,
            JsonAsyncTypesafeDomConsumerFactory.OK,
            session.listener().adjustContextGenerator(
                staffClient.httpClientContextGenerator()),
            new StaffCallback(session, new LinkedAccountsCallback(session)));
    }

    private static final class LinkedAccountsCallback
        extends AbstractProxySessionCallback<Map<Long, String>>
    {
        private LinkedAccountsCallback(final ProxySession session) {
            super(session);
        }

        @Override
        public void completed(final Map<Long, String> loginUidMap) {
            StringBuilderWriter sbw = new StringBuilderWriter();
            try (JsonWriter writer = JsonType.NORMAL.create(sbw)) {
                writer.startArray();
                for (Map.Entry<Long, String> loginUidPair
                    : loginUidMap.entrySet())
                {
                    writer.startObject();
                    writer.key("uid");
                    writer.value(loginUidPair.getKey());
                    writer.key(LOGIN);
                    writer.value(loginUidPair.getValue());
                    writer.endObject();
                }
                writer.endArray();
            } catch (IOException e) {
                failed(e);
                return;
            }

            session.response(HttpStatus.SC_OK, sbw.toString());
        }
    }

    private final class StaffCallback
        extends AbstractFilterFutureCallback<JsonObject, Map<Long, String>>
    {
        private final ProxySession session;

        private StaffCallback(
            final ProxySession session,
            final LinkedAccountsCallback callback)
        {
            super(callback);

            this.session = session;
        }

        @Override
        public void completed(final JsonObject result) {
            String linkedLogin = null;
            try {
                JsonMap root = result.asMap();
                JsonMap linkedLoginsMap = root.getMapOrNull("yandex");
                if (linkedLoginsMap != null) {
                    linkedLogin =
                        linkedLoginsMap.getString(LOGIN, null);
                }
            } catch (JsonException je) {
                failed(je);
                return;
            }

            session.logger().info(
                "Found linked login " + linkedLogin);
            if (linkedLogin != null) {
                BlackboxUserinfoRequest userInfoRequest =
                    new BlackboxUserinfoRequest(
                        BlackboxUserIdType.LOGIN,
                        linkedLogin);
                userInfoRequest.addHeader(
                    YandexHeaders.X_YA_SERVICE_TICKET,
                    webApi.blackboxTvm2Ticket());

                webApi.blackboxClient().userinfo(
                    userInfoRequest,
                    session.listener().adjustContextGenerator(
                        webApi.blackboxClient().httpClientContextGenerator()),
                    new BlackboxCallback(callback));
            } else {
                callback.completed(Collections.emptyMap());
            }
        }
    }

    private static final class BlackboxCallback
        extends AbstractFilterFutureCallback<
        List<BlackboxUserinfo>, Map<Long, String>>
    {
        private BlackboxCallback(
            final FutureCallback<? super Map<Long, String>> callback)
        {
            super(callback);
        }

        @Override
        public void completed(final List<BlackboxUserinfo> userinfos) {
            Map<Long, String> result = new LinkedHashMap<>();
            for (BlackboxUserinfo userinfo: userinfos) {
                result.put(userinfo.uid(), userinfo.login());
            }

            callback.completed(result);
        }
    }
}
