package ru.yandex.mail.search.web;

import java.io.IOException;

import org.apache.http.HttpException;
import org.apache.http.HttpStatus;

import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.EmptyAsyncConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.mail.search.web.info.ExtractChain;
import ru.yandex.mail.search.web.info.InfoExtractorFactory;
import ru.yandex.mail.search.web.info.PatternExtractChain;
import ru.yandex.ps.webtools.mail.MailSearchProject;
import ru.yandex.search.proxy.SearchProxyParams;

public class MailReindexHandler implements ProxyRequestHandler {
    private static final ExtractChain UID_CHAIN =
        new PatternExtractChain(
            InfoExtractorFactory.ONE_LONG,
            InfoExtractorFactory.UID_EXTRACT_CHAIN);

    private final MailSearchProject mailProject;

    public MailReindexHandler(final MailSearchProject mailProject) {
        this.mailProject = mailProject;
    }

    @Override
    public void handle(
        final ProxySession session)
        throws HttpException, IOException
    {
        String request = session.params().getString("request", null);
        Long uid = session.params().getLong(SearchProxyParams.PREFIX, null);
        if (uid == null) {
            uid = session.params().getLong("uid", null);
            if (uid == null && request != null) {
                String uidStr = UID_CHAIN.extract(request);
                if (uidStr != null) {
                    try {
                        uid = Long.parseLong(uidStr);
                    } catch (NumberFormatException nfe) {
                        throw new BadRequestException("Bad uid " + uidStr);
                    }
                }
            }
        }

        if (uid == null) {
            throw new BadRequestException("No uid found in request");
        }

        String defaultService = mailProject.config().reindexQueue();

        String service =
            session.params().getString(
                SearchProxyParams.SERVICE,
                defaultService);

        AsyncClient client = mailProject.reindexer(uid).adjust(session.context());
        final String uri =
            "/reindex?service=" + service + "&prefix=" + uid + "&webtools=true";
        session.logger().info("Reindexing " + uri);
        client.execute(
            mailProject.reindexerHost(uid),
            new BasicAsyncRequestProducerGenerator(uri),
            EmptyAsyncConsumerFactory.ANY_GOOD,
            new ReindexCallback(session, uid, service));
    }

    private static final class ReindexCallback
        extends AbstractProxySessionCallback<Void>
    {
        private final Long prefix;
        private final String service;

        private ReindexCallback(
            final ProxySession session,
            final Long prefix,
            final String service)
        {
            super(session);
            this.prefix = prefix;
            this.service = service;
        }

        @Override
        public void completed(final Void aVoid) {
            StringBuilderWriter sbw = new StringBuilderWriter();
            try (JsonWriter writer = JsonType.NORMAL.create(sbw)) {
                writer.startObject();
                writer.key(SearchProxyParams.PREFIX);
                writer.value(prefix);
                writer.key(SearchProxyParams.SERVICE);
                writer.value(service);
                writer.endObject();
            } catch (IOException e) {
                failed(e);
                return;
            }

            session.response(HttpStatus.SC_OK, sbw.toString());
        }
    }
}
