package ru.yandex.mail.search.web;

import java.io.IOException;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import org.apache.http.HttpException;

import ru.yandex.blackbox.BlackboxClient;
import ru.yandex.client.tvm2.Tvm2ServiceContextRenewalTask;
import ru.yandex.client.tvm2.Tvm2TicketRenewalTask;
import ru.yandex.collection.Pattern;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.request.RequestHandlerMapper;
import ru.yandex.json.parser.JsonException;
import ru.yandex.mail.search.web.config.ImmutablePsProjectConfig;
import ru.yandex.mail.search.web.config.ImmutableWebApiConfig;
import ru.yandex.mail.search.web.disk.serp.DiskImageHandler;
import ru.yandex.mail.search.web.disk.serp.DiskSerpHandler;
import ru.yandex.mail.search.web.iex.RemorphHandler;
import ru.yandex.search.proxy.universal.UniversalSearchProxy;

public class WebApi extends UniversalSearchProxy<ImmutableWebApiConfig> {
    private static final String SEARCH_CLIENT = "Search";

    private final AsyncClient diskSearchClient;

    private final AsyncClient mdsClient;

    private final AsyncClient staffClient;

    private final Tvm2TicketRenewalTask tvm2RenewTask;
    private final Tvm2TicketRenewalTask corpTvm2RenewTask;
    private final BlackboxClient blackboxClient;
    private final BlackboxClient corpBlackboxClient;

    private final List<WebtoolsProject> projects;

    public WebApi(final ImmutableWebApiConfig config) throws IOException {
        super(config);

        this.diskSearchClient =
            client("DiskSearch", config.diskSearch());

        this.mdsClient = client("MDS", config.mds());

        this.staffClient = client("Staff", config.staff());

        this.blackboxClient =
            registerClient(
                "Blackbox",
                new BlackboxClient(reactor(), config.blackbox()),
                config.blackbox());
        this.corpBlackboxClient =
            registerClient(
                "CorpBlackbox",
                new BlackboxClient(reactor(), config.corpBlackboxConfig()),
                config.corpBlackboxConfig());

        this.clients.remove(SEARCH_CLIENT);

        this.register(
            new Pattern<>("/disk/search", false),
            new DiskSerpHandler(this));
        this.register(
            new Pattern<>("/disk/image", false),
            new DiskImageHandler(this));
        this.register(
            new Pattern<>("/yandex/user/accounts", false),
            new LinkedAccountsHandler(this));
        this.register(
            new Pattern<>("/session/check", false),
            new SessionCheckHandler(this));

        this.register(
            new Pattern<>("/iex/remorph", false),
            new RemorphHandler(this),
            RequestHandlerMapper.POST);

        this.register(
            new Pattern<>("/support/onNewTicket", false),
            new NewStTicketHandler(this),
            RequestHandlerMapper.POST);

        this.register(
            new Pattern<>("/projects", false),
            new ProjectsHandler(this),
            RequestHandlerMapper.GET);

        try {
            tvm2RenewTask = new Tvm2TicketRenewalTask(
                logger.addPrefix("blackboxTvm2"),
                serviceContextRenewalTask(),
                config.blackboxTvm2());
            corpTvm2RenewTask = new Tvm2TicketRenewalTask(
                logger().addPrefix("corpBlackboxTvm2"),
                serviceContextRenewalTask(),
                config.corpBlackboxTvm2());
        } catch (JsonException | URISyntaxException | HttpException e) {
            throw new IOException(e);
        }

        List<WebtoolsProject> projects = new ArrayList<>();
        for (Map.Entry<String, ? extends ImmutablePsProjectConfig> prjConfig
            : config.projects().entrySet())
        {
            projects.add(
                prjConfig.getValue().projectType().project(
                    this,
                    prjConfig.getValue()));
        }

        this.projects = Collections.unmodifiableList(projects);
    }

    public AsyncClient diskSearchClient() {
        return diskSearchClient;
    }

    public AsyncClient mdsClient() {
        return mdsClient;
    }

    public AsyncClient staffClient() {
        return staffClient;
    }

    @Override
    public void start() throws IOException {
        super.start();

        tvm2RenewTask.start();
        corpTvm2RenewTask.start();

        for (WebtoolsProject project: projects) {
            project.start();
        }
    }

    @Override
    public void close() throws IOException {
        super.close();

        for (WebtoolsProject project: projects) {
            project.close();
        }

        tvm2RenewTask.cancel();
        corpTvm2RenewTask.cancel();
    }

    @Override
    public AsyncClient searchClient() {
        return searchClient;
    }

    public Tvm2ServiceContextRenewalTask serviceContextRenewalTask() {
        return serviceContextRenewalTask;
    }

    public Tvm2TicketRenewalTask tvm2RenewTask() {
        return tvm2RenewTask;
    }

    public Tvm2TicketRenewalTask corpTvm2RenewTask() {
        return corpTvm2RenewTask;
    }

    public BlackboxClient blackboxClient() {
        return blackboxClient;
    }

    public BlackboxClient corpBlackboxClient() {
        return corpBlackboxClient;
    }

    public String blackboxTvm2Ticket() {
        return tvm2RenewTask.ticket();
    }

    public String corpBlackboxTvm2Ticket() {
        return corpTvm2RenewTask.ticket();
    }

    public List<WebtoolsProject> projects() {
        return projects;
    }

    public void register(
        final ImmutablePsProjectConfig project,
        final String path,
        final ProxyRequestHandler handler)
    {
        register(
            new Pattern<>('/' + project.projectId() + path, false),
            handler);
    }

    public void register(
        final ImmutablePsProjectConfig project,
        final String path,
        final ProxyRequestHandler handler,
        final String... method)
    {
        register(
            new Pattern<>('/' + project.projectId() + path, false),
            handler,
            method);
    }
}
