package ru.yandex.mail.search.web.config;

import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;

import ru.yandex.http.config.HttpTargetConfig;
import ru.yandex.http.config.HttpTargetConfigBuilder;
import ru.yandex.mail.search.web.config.check.MetricType;
import ru.yandex.mail.search.web.health.base.PsProjectType;
import ru.yandex.mail.search.web.health.update.MetricUpdateTaskFactory;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.parser.searchmap.SearchMapConfig;
import ru.yandex.parser.searchmap.SearchMapConfigBuilder;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.EnumParser;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.search.mop.manager.config.MopManagerConfig;
import ru.yandex.search.mop.manager.config.MopManagerConfigBuilder;

public abstract class AbstractPsProjectConfigBuilder
    <T extends AbstractPsProjectConfigBuilder<T>>
     implements PsProjectConfig
{
    private SearchMapConfigBuilder searchMapConfig;
    private MopManagerConfigBuilder mopConfig;
    private List<MetricUpdateTaskFactory> metrics;
    private HttpTargetConfigBuilder searchConfig;
    private String defaultService;
    private int metricsUpdateWorkers;
    private String name;
    private String description;
    private String projectId;
    private String luceneDropPassword;
    private PsProjectType projectType;
    private int luceneShards;
    private int totalShardsCount;

    private String yasmProject;
    private String yasmLuceneTemplate;

    protected AbstractPsProjectConfigBuilder(final PsProjectConfig config) {
        searchMapConfig(config.searchMapConfig());
        metrics(config.metrics());
        searchConfig(config.searchConfig());
        defaultService(config.defaultService());
        metricsUpdateWorkers(config.metricsUpdateWorkers());

        name(config.name());
        projectId(config.projectId());
        description(config.description());
        projectType(config.projectType());
        luceneShards(config.luceneShards());
        totalShardsCount(config.totalShardsCount());
        luceneDropPassword(config.luceneDropPassword());

        yasmProject(config.yasmProject());
        yasmLuceneTemplate(config.yasmLuceneTemplate());
        mopConfig(config.mopConfig());
    }

    protected AbstractPsProjectConfigBuilder(
        final IniConfig config,
        final PsProjectConfig defaults)
        throws ConfigException
    {
        this.searchMapConfig =
            new SearchMapConfigBuilder(
                config.section("searchmap"),
                defaults.searchMapConfig());
        metrics = new ArrayList<>();
        projectType =
            config.getEnum(PsProjectType.class, "type");

        Set<String> metricsNames =
            config.getAll(
                "metrics",
                Collections.emptySet(),
                new CollectionParser<>(
                    NonEmptyValidator.TRIMMED,
                    LinkedHashSet::new));
        for (String name : metricsNames) {
            IniConfig section =
                config.sectionOrNull(name);
            if (section == null) {
                metrics.add(
                    new EnumParser<>(MetricType.class)
                        .apply(name).create(null));
            } else {
                metrics.add(
                    section.getEnum(MetricType.class, "type").create(section));
            }
        }
        this.searchConfig =
            new HttpTargetConfigBuilder(
                config.section("search"),
                defaults.searchConfig());
        this.defaultService =
            config.getString("default-service", defaults.defaultService());
        this.metricsUpdateWorkers =
            config.getInt(
                "metrics-update-workers",
                defaults.metricsUpdateWorkers());
        this.name = config.getString("name", defaults.name());
        this.description =
            config.getString("description", defaults.description());
        this.projectId = config.getString("project_id", defaults.projectId());
        this.luceneShards =
            config.getInt("lucene-shards", defaults.luceneShards());
        this.totalShardsCount =
            config.getInt("total-shards-count", defaults.totalShardsCount());

        this.luceneDropPassword = config.getString(
            "lucene-drop-password",
            defaults.luceneDropPassword());

        this.yasmProject =
            config.getString("yasm-project", defaults.yasmProject());
        this.yasmLuceneTemplate =
            config.getString("yasm-lucene-template", defaults.yasmLuceneTemplate());

        IniConfig mopSection = config.sectionOrNull("mop");
        if (mopSection != null) {
            mopConfig = new MopManagerConfigBuilder(mopSection, defaults.mopConfig());
        } else {
            mopConfig = null;
        }
    }

    @Override()
    public SearchMapConfigBuilder searchMapConfig() {
        return searchMapConfig;
    }

    public T searchMapConfig(final SearchMapConfig value) {
        this.searchMapConfig = new SearchMapConfigBuilder(
            value);
        return self();
    }

    public List<MetricUpdateTaskFactory> metrics() {
        return metrics;
    }

    public T metrics(final List<MetricUpdateTaskFactory> metrics) {
        this.metrics = new ArrayList<>(
            metrics);
        return self();
    }

    public abstract T self();

    public ImmutablePsProjectConfig build() throws ConfigException {
        return new ImmutablePsProjectConfig(
            this);
    }

    @Override()
    public HttpTargetConfigBuilder searchConfig() {
        return searchConfig;
    }

    public T searchConfig(final HttpTargetConfig value) {
        this.searchConfig = new HttpTargetConfigBuilder(
            value);
        return self();
    }

    @Override()
    public String defaultService() {
        return defaultService;
    }

    public T defaultService(final String value) {
        this.defaultService = value;
        return self();
    }

    @Override()
    public int metricsUpdateWorkers() {
        return metricsUpdateWorkers;
    }

    public T metricsUpdateWorkers(final int value) {
        this.metricsUpdateWorkers = value;
        return self();
    }

    @Override()
    public String name() {
        return name;
    }

    public T name(final String value) {
        this.name = value;
        return self();
    }

    @Override()
    public String description() {
        return description;
    }

    public T description(final String value) {
        this.description = value;
        return self();
    }

    @Override()
    public String projectId() {
        return projectId;
    }

    public T projectId(final String value) {
        this.projectId = value;
        return self();
    }

    @Override
    public PsProjectType projectType() {
        return projectType;
    }

    public T projectType(final PsProjectType projectType) {
        this.projectType = projectType;
        return self();
    }

    @Override
    public int luceneShards() {
        return luceneShards;
    }

    public T luceneShards(final int luceneShards) {
        this.luceneShards = luceneShards;
        return self();
    }

    @Override
    public String luceneDropPassword() {
        return luceneDropPassword;
    }

    public T luceneDropPassword(final String password) {
        this.luceneDropPassword = password;
        return self();
    }

    @Override
    public String yasmProject() {
        return yasmProject;
    }

    public T yasmProject(final String yasmProject) {
        this.yasmProject = yasmProject;
        return self();
    }

    @Override
    public String yasmLuceneTemplate() {
        return yasmLuceneTemplate;
    }

    public T yasmLuceneTemplate(final String yasmLuceneTemplate) {
        this.yasmLuceneTemplate = yasmLuceneTemplate;
        return self();
    }

    @Override
    public int totalShardsCount() {
        return totalShardsCount;
    }

    public T totalShardsCount(final int count) {
        this.totalShardsCount = count;
        return self();
    }

    @Override
    public MopManagerConfigBuilder mopConfig() {
        return mopConfig;
    }

    public T mopConfig(
        final MopManagerConfig mopConfig)
    {
        if (mopConfig != null) {
            this.mopConfig = new MopManagerConfigBuilder(mopConfig);
        } else {
            this.mopConfig = null;
        }

        return self();
    }
}
