package ru.yandex.mail.search.web.disk.serp;

import java.io.IOException;
import java.io.OutputStreamWriter;
import java.nio.charset.Charset;
import java.nio.charset.CodingErrorAction;
import java.util.ArrayList;
import java.util.List;

import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NByteArrayEntity;
import org.apache.http.nio.protocol.BasicAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HttpContext;

import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.BasicProxySession;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.CharsetUtils;
import ru.yandex.http.util.FilterFutureCallback;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.NByteArrayEntityFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.io.DecodableByteArrayOutputStream;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.mail.search.web.WebApi;
import ru.yandex.parser.uri.QueryConstructor;

public class DiskSerpHandler implements HttpAsyncRequestHandler<HttpRequest> {
    private static final String STID = "stid";
    private static final String KPS = "kps";
    private static final String IMG = "img";
    private static final String MODIFICATION = "modification";
    private static final String MODIFICATION_PLUS = "plus";
    private static final String MODIFICATION_MINUS = "minus";
    private static final String IMG_PREFIX = "/api/disk/image?stid=";

    private final WebApi proxy;

    public DiskSerpHandler(final WebApi proxy) {
        this.proxy = proxy;
    }

    @Override
    public HttpAsyncRequestConsumer<HttpRequest> processRequest(
        final HttpRequest request,
        final HttpContext context)
        throws HttpException, IOException
    {
        return new BasicAsyncRequestConsumer();
    }

    @Override
    public void handle(
        final HttpRequest request,
        final HttpAsyncExchange exchange,
        final HttpContext context)
        throws HttpException, IOException
    {
        BasicProxySession session =
            new BasicProxySession(proxy, exchange, context);
        QueryConstructor qc =
            new QueryConstructor(proxy.config().diskSearch().host() + "/?");
        qc.append(KPS, session.params().getLong(KPS));
        qc.append("text", session.params().getString("request"));
        qc.append("only", "id,stid");
        qc.append("source", "mail_search_webtools");
        AsyncClient client = proxy.diskSearchClient().adjust(session.context());

        MultiFutureCallback<JsonObject> mfcb =
            new MultiFutureCallback<>(
                new DiskProxyCallback(
                    session,
                    CharsetUtils.acceptedCharset(request)));

        String i2t = qc.toString() + "&i2t-only";
        client.execute(
            proxy.config().diskSearch().host(),
            new BasicAsyncRequestProducerGenerator(i2t),
            JsonAsyncTypesafeDomConsumerFactory.OK,
            session.listener().adjustContextGenerator(
                client.httpClientContextGenerator()),
            new DiskWrapperCallback(
                i2t,
                session.logger(),
                mfcb.newCallback()));

        String oldCv = qc.toString() + "&old-cv-only";
        client.execute(
            proxy.config().diskSearch().host(),
            new BasicAsyncRequestProducerGenerator(oldCv),
            JsonAsyncTypesafeDomConsumerFactory.OK,
            session.listener().adjustContextGenerator(
                client.httpClientContextGenerator()),
            new DiskWrapperCallback(
                oldCv,
                session.logger(),
                mfcb.newCallback()));

        mfcb.done();
    }

    private static final class DiskWrapperCallback
        extends FilterFutureCallback<JsonObject>
    {
        private final PrefixedLogger logger;
        private final String uri;

        private DiskWrapperCallback(
            final String uri,
            final PrefixedLogger logger,
            final FutureCallback<? super JsonObject> callback)
        {
            super(callback);

            this.uri = uri;
            this.logger = logger;
        }

        @Override
        public void completed(final JsonObject result) {
            logger.info(uri + " completed " + JsonType.NORMAL.toString(result));
            super.completed(result);
        }
    }

    private static final class DiskProxyCallback
        extends AbstractProxySessionCallback<List<JsonObject>>
    {
        private final Charset acceptedCharset;
        private final JsonType jsonType;

        private DiskProxyCallback(
            final ProxySession session,
            final Charset acceptedCharset)
            throws BadRequestException
        {
            super(session);
            this.acceptedCharset = acceptedCharset;
            this.jsonType = JsonTypeExtractor.NORMAL.extract(session.params());
        }

        private List<String> fetchProperties(
            final JsonObject serp)
            throws JsonException
        {
            JsonList results =
                serp.asMap().getMap("response").getList("results");
            List<String> out = new ArrayList<>(results.size());
            for (JsonObject result: results) {
                for (JsonObject group: result.asMap().getList("groups")) {
                    for (JsonObject doc
                        : group.asMap().getList("documents"))
                    {
                        out.add(
                            doc.asMap().getMap("properties").getString(STID));
                    }
                }
            }

            return out;
        }

        private void modification(
            final JsonWriter writer,
            final int index,
            final int otherIndex)
            throws IOException
        {
            if (otherIndex < 0) {
                return;
            }
            if (otherIndex > index) {
                writer.key(MODIFICATION);
                writer.value(MODIFICATION_MINUS);
            } else if (otherIndex < index) {
                writer.key(MODIFICATION);
                writer.value(MODIFICATION_PLUS);
            }
        }

        private void write(
            final JsonWriter writer,
            final JsonObject i2tSerp,
            final JsonObject oldCvSerp)
            throws JsonException, IOException
        {
            List<String> i2t = fetchProperties(i2tSerp);
            List<String> oldCv = fetchProperties(oldCvSerp);
            session.logger().info(
                "I2T " + i2t.size() + " OldCv " + oldCv.size());
            writer.startArray();
            for (int i = 0; i < Math.max(i2t.size(), oldCv.size()); i++) {
                writer.startObject();
                if (i < i2t.size()) {
                    String stid = i2t.get(i);

                    writer.key("i2t");
                    writer.startObject();
                    writer.key(IMG);
                    writer.value(IMG_PREFIX + stid);
                    modification(writer, i, oldCv.indexOf(stid));
                    writer.endObject();
                }

                if (i < oldCv.size()) {
                    String stid = oldCv.get(i);
                    writer.key("oldCv");
                    writer.startObject();
                    writer.key(IMG);
                    writer.value(IMG_PREFIX + stid);
                    modification(writer, i, i2t.indexOf(stid));
                    writer.endObject();
                }

                writer.endObject();
            }

            writer.endArray();
        }

        @Override
        public void completed(final List<JsonObject> serps) {
            DecodableByteArrayOutputStream out =
                new DecodableByteArrayOutputStream();
            try (OutputStreamWriter outWriter =
                     new OutputStreamWriter(
                         out,
                         acceptedCharset.newEncoder()
                             .onMalformedInput(CodingErrorAction.REPLACE)
                             .onUnmappableCharacter(CodingErrorAction.REPLACE));
                 JsonWriter writer = jsonType.create(outWriter))
            {
                write(writer, serps.get(0), serps.get(1));
            } catch (JsonException | IOException e) {
                failed(e);
                return;
            }

            NByteArrayEntity entity =
                out.processWith(NByteArrayEntityFactory.INSTANCE);
            entity.setContentType(
                ContentType.APPLICATION_JSON.withCharset(acceptedCharset)
                    .toString());
            session.response(HttpStatus.SC_OK, entity);
        }
    }
}
