package ru.yandex.mail.search.web.drop;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;

import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.StringEntity;

import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.EmptyAsyncConsumerFactory;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.mail.search.web.DefaultPsProject;
import ru.yandex.mail.search.web.health.HealthCheckService;
import ru.yandex.mail.search.web.health.base.ProjectQueue;
import ru.yandex.mail.search.web.health.base.PropertyStatus;
import ru.yandex.mail.search.web.health.base.Shard;
import ru.yandex.mail.search.web.health.base.ShardGroup;
import ru.yandex.mail.search.web.health.base.ShardReplica;
import ru.yandex.parser.uri.QueryConstructor;

public class DropLuceneHostHandler
    extends AbstractDropHandler
    implements ProxyRequestHandler
{
    private final HealthCheckService healthService;
    private final DefaultPsProject project;

    public DropLuceneHostHandler(final HealthCheckService healthService) {
        this.healthService = healthService;
        this.project = healthService.project();
    }

    @Override
    public void handle(
        final ProxySession session)
        throws HttpException, IOException
    {
        String host = session.params().getString("host");

        List<ShardDroppableCheckResult> shards = new ArrayList<>();

        Set<String> queues = new LinkedHashSet<>();
        Set<Integer> basePorts = new LinkedHashSet<>();
        for (ProjectQueue queue
            : healthService.projectRoot().queues().values())
        {
            List<ShardGroup> groups = queue.shardsByHost(host);
            for (ShardGroup group: groups) {
                for (Shard shard: group.shards()) {
                    for (ShardReplica directReplica: shard.replicas().values()) {
                        if (host.equalsIgnoreCase(directReplica.host().hostname())) {
                            basePorts.add(directReplica.host().basePort());
                            shards.add(calcShard(shard, directReplica));
                            queues.add(queue.service());
                        }
                    }
                }
            }
        }

        PropertyStatus worstStatus = PropertyStatus.OK;

        for (ShardDroppableCheckResult result: shards) {
            if (result.status().ordinal() > worstStatus.ordinal()) {
                worstStatus = result.status();
            }
        }

        boolean drop = session.params().getBoolean("drop", false);
        if (drop) {
            if (basePorts.size() > 1) {
                throw new BadRequestException(
                    "Multi containers on one host are not supported by now "
                        + basePorts);
            }

            if (worstStatus == PropertyStatus.OK
                || worstStatus == PropertyStatus.WARNING)
            {
                dropHost(
                    new DropContext(
                        host,
                        basePorts.iterator().next(),
                        shards,
                        project,
                        session,
                        queues));
                return;
            } else {
                session.logger().warning("Bad status");
            }
        }

        StringBuilderWriter sbw = new StringBuilderWriter();
        try (JsonWriter writer = JsonType.HUMAN_READABLE.create(sbw)) {
            writer.startObject();
            writer.key("status");
            writer.value(worstStatus.name());
            writer.key("shards");
            writer.startArray();
            for (ShardDroppableCheckResult result: shards) {
                writer.value(result);
            }
            writer.endArray();
            writer.endObject();
        }

        session.response(HttpStatus.SC_OK, new StringEntity(
            sbw.toString(),
            StandardCharsets.UTF_8));
    }

    protected void dropHost(final DropContext context) {
        context.session().logger().warning(
            "Dropping host " + context.hostname());
        FutureCallback<Object> callback = new DropCallback(context);
        QueryConstructor qc = new QueryConstructor("/dropp-inndex?");
        try {
            qc.append("password", project.config().luceneDropPassword());
        } catch (BadRequestException bre) {
            callback.failed(bre);
            return;
        }

        BasicAsyncRequestProducerGenerator generator =
            new BasicAsyncRequestProducerGenerator(qc.toString());

        generator.addHeader(
            "ticket",
            project.config().luceneDropPassword());

        context.client().execute(
            context.indexerHost(),
            generator,
            EmptyAsyncConsumerFactory.ANY_GOOD,
            context.session().listener().createContextGeneratorFor(
                context.client()),
            callback);
    }

    private static class DropCallback
        extends AbstractProxySessionCallback<Object>
    {
        private final DropContext context;

        public DropCallback(final DropContext context) {
            super(context.session());

            this.context = context;
        }

        @Override
        public void completed(final Object o) {
            session.logger().warning("Dropped");
            StringBuilderWriter sbw = new StringBuilderWriter();
            try (JsonWriter writer = JsonType.HUMAN_READABLE.create(sbw)) {
                writer.startObject();
                writer.key("status");
                writer.value(PropertyStatus.OK);
                writer.key("host");
                writer.value(context.hostname());
                writer.endObject();
            } catch (IOException ioe) {
                failed(ioe);
            }

            session.response(HttpStatus.SC_OK, new StringEntity(
                sbw.toString(),
                StandardCharsets.UTF_8));
        }
    }
}
